package org.lsst.ccs.drivers.keithley;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the Keithley N6517 device driver
 *     (based on Owen Saxton's Agilent driver tester)
 *
 *  @author Homer Neal
 */
public class TestN6517 extends TestScpi {

    private final N6517 n65;

    public enum onOff {
        OFF, ON;
    }


   /**
    *  Constructor
    */
    public TestN6517()
    {
        super(new N6517());
        n65 = (N6517)scpi;
    }


   /**
    *  Opens connection to a device using all defaults
    */
    @Command(name="open", description="Open connection to device using defaults")
    public void open() throws DriverException
    {
        n65.open();
    }


   /**
    *  Opens connection to a device.
    */
    @Command(name="open", description="Open connection to device")
    public String open(@Argument(name="host", description="Host name")
                     String host
                     ) throws DriverException
    {
        n65.open(host);
        return n65.printdevid();
    }


   /**
    *  Opens connection to a device.
    */
    @Command(name="open", description="Open connection to device")
    public String open(@Argument(name="host", description="Host name")
                     String host,
                     @Argument(name="port", description="Port number")
                     int port
                     ) throws DriverException
    {
        n65.open(host, port);
        return n65.printdevid();
    }


   /**
    *  Opens FTDI connection to a device.
    */
    @Command(name="openftdi", description="Open FTDI connection to device")
    public String openftdi(@Argument(name="serialname", description="Serial device name")
                     String host,
                     @Argument(name="baud", description="baud rate")
                     int port
                     ) throws DriverException
    {
        n65.openftdi(host, port);
        return n65.printdevid();
    }


   /**
    *  Sets the rate
    */
    @Command(name="setrate", description="Set read rate")
    public void setRate(@Argument(name="value", description="number of power cycles")
                           int value) throws DriverException
    {
        n65.setRate(value);
    }


   /**
    *  Sets ARM count
    */
    @Command(name="setarmcount", description="Set the ARM count")
    public void setArmCount(@Argument(name="value", description="ARM count")
                           int value) throws DriverException
    {
        n65.setArmCount(value);
    }


   /**
    *  Sets TRIGger count
    */
    @Command(name="settrigcount", description="Set the trigger count")
    public void setTrigCount(@Argument(name="value", description="trigger count")
                           int value) throws DriverException
    {
        n65.setTrigCount(value);
    }


   /**
    *  Clear buffer
    */
    @Command(name="clrbuff", description="Clear the read buffer")
    public void clearBuffer() throws DriverException
    {
        n65.clrBuff();
    }


   /**
    *  Read buffer
    */
    @Command(name="readbuff", description="read the buffer")
    public String readBuffer() throws DriverException
    {
        return "Buffer = " + getString(n65.readBuffer()) ;
    }


   /**
    *  Reset device to a useable state
    */
    @Command(name="reset", description="reset the device to a useable state")
    public void reset() throws DriverException
    {
        n65.reset() ;
    }


   /**
    *  Set buffer size
    */
    @Command(name="setbuffsize", description="Set the buffer size.")
    public void setBuffSize(@Argument(name="value", description="buffer size")
                           int value) throws DriverException
    {
        n65.setBuffSize(value);
    }


   /**
    *  Sets the voltage range
    */
    @Command(name="setvoltagerange", description="Set the voltage range. Can be 10V, 50V or 500V")
    public void setVoltageRange(@Argument(name="value", description="Voltage Range to set")
                           double value) throws DriverException
    {
        n65.setVoltageRange(value);
    }


   /**
    *  Sets the voltage.
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        n65.setVoltage(value);
    }


   /**
    *  Ramps to the desired voltage.
    */
    @Command(name="rampvolts", description="Set the voltage")
    public void rampVolts(@Argument(name="duration", description="number of second for the ramp")
                           double duration,
                           @Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        n65.rampVolts(duration,value);
    }


   /**
    *  Shows the set voltage.
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String getVoltage() throws DriverException
    {
        return "Voltage = " + n65.getVoltage();
    }


   /**
    *  Reads the actual voltage.
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + n65.readVoltage();
    }


   /**
    *  Sets the soft voltage limit.
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value) throws DriverException
    {
        n65.setVoltageLimit(value);
    }


   /**
    *  Shows the soft voltage limit.
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + n65.getVoltageLimit();
    }


   /**
    *  Sets the current range
    */
    @Command(name="setcurrentrange", description="Set the current range")
    public void setCurrentRange(@Argument(name="value", description="Current Range to set")
                           double value) throws DriverException
    {
        n65.setCurrentRange(value);
    }


   /**
    *  Sets the current.
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        n65.setCurrent(value);
    }


   /**
    *  Zero correct the current
    */
    @Command(name="zerocorrectcurrent", description="Zero correct the current")
    public void zeroCorrectCurrent() throws DriverException
    {
        n65.zeroCorrectCurrent();
    }


   /**
    *  Shows the set current.
    */
    @Command(name="showcurrent", description="Show the set current")
    public String getCurrent() throws DriverException
    {
        return "Current = " + n65.getCurrent();
    }


   /**
    *  Reads the actual current.
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + n65.readCurrent();
    }


   /**
    *  Sets the soft current limit.
    */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value",
                                          description="Current limit to set")
                                double value) throws DriverException
    {
        n65.setCurrentLimit(value);
    }


   /**
    *  Shows the soft current limit.
    */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public String getCurrentLimit() throws DriverException
    {
        return "Current limit = " + n65.getCurrentLimit();
    }


   /**
    *  Turns the output on or off.
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          onOff state) throws DriverException
    {
        n65.setOutput(state == onOff.ON);
    }


   /**
    *  Shows the output state.
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(n65.getOutput());
    }


   /**
    *  Converts an array of numbers to a string.
    */
    private StringBuilder getString(double[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }


   /**
    *  Converts a boolean to on/off.
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
