package org.lsst.ccs.drivers.keithley;

import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Routines for controlling a Keithley model 2231A power supply
 *
 *  @author  Owen Saxton
 */
public class Model2231A extends Scpi implements PowerSupplyDriver {

    public static final int
        BAUDRATE     = 9600,
        MIN_CHANNEL  = 1,
        MAX_CHANNEL  = 3,
        NUM_CHANNELS = MAX_CHANNEL - MIN_CHANNEL + 1;
    public static final double MAX_VALUE = Double.POSITIVE_INFINITY;

    private final Timer timer = new Timer("Timer_2231A", true);
    private final long[]
        onDelay = new long[NUM_CHANNELS],
        offDelay = new long[NUM_CHANNELS];
    private final SetOutput[] outTask = new SetOutput[NUM_CHANNELS];


    /**
     *  Timer task to implement the delayed setting of the output state.
     */
    class SetOutput extends TimerTask {

        private final int chan;
        private final String command;

        SetOutput(String command, int chan)
        {
            this.command = command;
            this.chan = chan;
        }
    
        @Override
        public void run()
        {
            try {
                writeChan(command, chan);
            }
            catch (DriverException e) {
            }
            outTask[chan - MIN_CHANNEL] = null;
        }
    }


    /**
     *  Constructor.
     */
    public Model2231A() {
        setOptions(Option.NO_NET);   // Disallow network connections
        setDefaultParm(BAUDRATE);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate, ignored - always set to 9600
     *  @param  commParm  The communications parameters, ignored - always set to 0
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int commParm) throws DriverException
    {
        super.open(connType, ident, BAUDRATE, 0);
        setTerminator(Terminator.LF);
        try {
            write("SYST:REM");    // Enable full remote operation
            checkIdentification("Keithley", CHECK_STARTS_WITH, "2231A", CHECK_STARTS_WITH);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


    /**
     *  Turns output on or off for a channel.
     *
     *  @param  on    Turn on if true, off if false
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        checkChannel(chan);
        int index = chan - MIN_CHANNEL;
        long delay = on ? onDelay[index] : offDelay[index];
        String command = "CHAN:OUTP " + (on ? "1" : "0");
        SetOutput task = outTask[index];
        if (task != null) {
            task.cancel();
            outTask[index] = null;
        }
        if (delay == 0) {
            writeChan(command, chan);
        }
        else {
            outTask[index] = new SetOutput(command, chan);
            timer.schedule(outTask[index], delay);
        }
    }


    /**
     *  Gets the output state of a channel.
     *
     *  @param  chan  The channel number
     *  @return  The output state
     *  @throws  DriverException
     */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return readIntegerChan("CHAN:OUTP?", chan) != 0;
    }


    /**
     *  Sets the power-on delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        onDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


    /**
     *  Sets the power-off delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        offDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeChan("VOLT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Sets the voltage for all channels.
     *
     *  @param  values  The three-element array of values to set
     *  @throws  DriverException
     */
    public void setVoltage(double[] values) throws DriverException
    {
        writeCommand("APP:VOLT " + makeString(values));
    }


    /**
     *  Gets the set voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT?", chan);
    }


    /**
     *  Gets the set voltages for all channels.
     *
     *  @return  A three-element array of voltages
     *  @throws  DriverException
     */
    public double[] getVoltage() throws DriverException
    {
        return readDoubleArray("APP:VOLT?");
    }


    /**
     *  Gets the maximum voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The maximum voltage
     *  @throws  DriverException
     */
    public double getMaximumVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT? MAX", chan);
    }


    /**
     *  Reads the voltage for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:VOLT?", chan);
    }


   /**
     *  Sets the current for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeChan("CURR " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Sets the current for all channels.
     *
     *  @param  values  The three-element array of values to set
     *  @throws  DriverException
     */
    public void setCurrent(double[] values) throws DriverException
    {
        writeCommand("APP:CURR " + makeString(values));
    }


    /**
     *  Gets the set current for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR?", chan);
    }


    /**
     *  Gets the set current for all channels.
     *
     *  @return  The three-element array of currents
     *  @throws  DriverException
     */
    public double[] getCurrent() throws DriverException
    {
        return readDoubleArray("APP:CURR?");
    }


    /**
     *  Gets the maximum current for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The maximum current
     *  @throws  DriverException
     */
    public double getMaximumCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR? MAX", chan);
    }


    /**
     *  Reads the current for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:CURR?", chan);
    }


    /**
     *  Sets the voltage limit for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    public void setVoltageLimit(double value, int chan) throws DriverException
    {
        writeChan("VOLT:LIMIT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Gets the voltage limit for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDoubleChan("VOLT:LIMIT?", chan);
    }


    /**
     *  Turns on or off the voltage limit check for a channel.
     *
     *  @param  on    Whether to turn on the limit check
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    public void setVoltageLimitCheck(boolean on, int chan) throws DriverException
    {
        writeChan("VOLT:LIMIT:STATE " + (on ? 1 : 0), chan);
    }


    /**
     *  Gets whether voltage limit check is enabled for a channel.
     *
     *  @param  chan  The channel number
     *  @return  Whether voltage limit check is enabled
     *  @throws  DriverException
     */
    public boolean getVoltageLimitCheck(int chan) throws DriverException
    {
        return readIntegerChan("VOLT:LIMIT:STATE?", chan) == 1;
    }


    /**
     *  Locks or unlocks the front panel.
     *
     *  @param  lock  True to lock the panel; false to unlock it
     *  @throws  DriverException
     */
    public void lockPanel(boolean lock) throws DriverException
    {
        writeCommand(lock ? "SYST:RWL" : "SYST:REM");
    }


    /**
     *  Checks a channel for validity.
     */
    private void checkChannel(int chan) throws DriverException
    {
        if (chan < MIN_CHANNEL || chan > MAX_CHANNEL) {
            throw new DriverException("Invalid channel number");
        }
    }


    /**
     *  Writes a channel select command.
     */
    private void writeSelect(int chan) throws DriverException
    {
        checkChannel(chan);
        writeCommand("INST:NSEL " + chan);
    }


    /**
     *  Writes a command to a channel.
     */
    private synchronized void writeChan(String instr, int chan) throws DriverException
    {
        writeSelect(chan);
        writeCommand(instr);
    }


    /**
     *  Writes a command to a channel and reads the double result.
     */
    private synchronized double readDoubleChan(String instr, int chan) throws DriverException
    {
        writeSelect(chan);
        return readDouble(instr);
    }


    /**
     *  Writes a command to a channel and reads the integer result.
     */
    private synchronized int readIntegerChan(String instr, int chan) throws DriverException
    {
        writeSelect(chan);
        return readInteger(instr);
    }


    /**
     *  Makes a comma-separated string from an array of doubles.
     */
    private static String makeString(double[] values) throws DriverException
    {
        if (values.length != MAX_CHANNEL) {
            throw new DriverException("Invalid value array size");
        }
        StringBuilder str = new StringBuilder();
        for (int j = 0; j < values.length; j++) {
            if (j != 0) {
                str.append(",");
            }
            str.append(Double.isFinite(values[j]) ? values[j] : "MAX");
        }

        return str.toString();
    }

}
