package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Class for accessing the LTC2945 chip
 *
 *  @author  Owen Saxton
 */
public class LTC2945 {

    private final int
        ADC_INCREMENT   = 0x400,
        ADC_CONTROL     = 0,
        ADC_ALERT       = 1,
        ADC_STATUS      = 2,
        ADC_FAULT       = 3,
        ADC_FAULT_COR   = 4,
        ADC_POWER_BASE  = 5,
        ADC_SENSE_BASE  = 10,
        ADC_V_IN_BASE   = 15,
        ADC_AD_IN_BASE  = 20,
        ADC_CURR_MEAS   = 0,
        ADC_MAX_MEAS    = 1,
        ADC_MIN_MEAS    = 2,
        ADC_MAX_THRESH  = 3,
        ADC_MIN_THRESH  = 4;

    private static final double
        VOLT_SCALE = 102.4 / 4096,
        CURR_SCALE = 0.1024 / 4096;

    private final Srp srp;
    private final int adcBase;


    /**
     *  Constructor.
     *
     *  @param  srp      The srp object to use
     *  @param  adcBase  The LTC2945 base address 
     */
    public LTC2945(Srp srp, int adcBase)
    {
        this.srp = srp;
        this.adcBase = adcBase;
    }


    /**
     *  Reads a channel's voltage
     *
     *  @param  chan  The channel number
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readVoltage(int chan) throws DriverException
    {
        return srp.readReg(adcBase + ADC_INCREMENT * chan + ADC_V_IN_BASE + ADC_CURR_MEAS) * VOLT_SCALE;
    }


    /**
     *  Reads a channel's current
     *
     *  @param  chan  The channel number
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readCurrent(int chan) throws  DriverException
    {
        return srp.readReg(adcBase + ADC_INCREMENT * chan + ADC_SENSE_BASE + ADC_CURR_MEAS) * CURR_SCALE;
    }

}
