package org.lsst.ccs.drivers.apcpdu;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.commons.DriverException;
import java.util.logging.Logger;
import java.util.logging.Level;

/**
 *  Driver for controlling an APC UPS with an AP963x network card
 *
 *  @author Owen Saxton
 */
public class AP9630UPS extends Session {

    /**
     *  Public constants
     */
    public static final int
        ST_ONLINE = 0,    // Status - online
        ST_BATTERY = 1,   // Status - on battery
        ST_OFFLINE = 2;   // Status - offline

    /**
     *  Private data.
     */
    private static final int
        DEFAULT_BAUDRATE = 9600,
        LOG_TIMEOUT = 500,
        RUN_TIMEOUT = 50;
    private String product, fwVersion;
    private static final Logger logger = Logger.getLogger(AP9630UPS.class.getName());


    /**
     *  Constructor
     */
    public AP9630UPS()
    {
        super(OPTN_KEEP_ALIVE, "apc>", "User Name :", "Password  :", null, Ascii.Terminator.CR);
    }


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @param  username  The user name
     *  @param  password  The password
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident, String username, String password) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, username, password, LOG_TIMEOUT, RUN_TIMEOUT);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: TELNET or SERIAL
     *  @param  ident     The host name (TELNET) or port name (SERIAL)
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, RUN_TIMEOUT);
    }


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
        try {
            super.close();
        }
        finally {
            product = null;
            fwVersion = null;
        }
    }


    /**
     *  Gets the product name
     *
     *  @return  The product name
     *  @throws  DriverException
     */
    public String getProductName() throws DriverException
    {
        if (product == null) {
            getVersionInfo();
        }
        return product;
    }


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFWVersion() throws DriverException
    {
        if (fwVersion == null) {
            getVersionInfo();
        }
        return fwVersion;
    }


    /**
     *  Gets the UPS status
     *
     *  @return  The status: ONLINE, BATTERY...
     *  @throws  DriverException
     */
    /*
    public int getStatus() throws DriverException
    {
        String[] reply = receiveString("detstatus -ss");
        checkReplyLength(reply, 6);
        String[] words = getStringValue(reply[0]).split(" ");
        return words[0].equals("Online") ? ST_ONLINE : ST_BATTERY;  // Other possibilities???
    }
    */
    public int getStatus() throws DriverException
    {
        String[] reply = receiveString("detstatus -ss");
        checkReplyLength(reply, 6);
        String[] words = getStringValue(reply[0]).split(" ");
        boolean isOnline = words[0].equals("Online");
        if  ( !isOnline ) {
              logger.log(Level.INFO,"UPS detstatus -ss reply:\n {0}\n {1}\n {2}\n {3}\n {4}\n {5}",reply);
              return ST_BATTERY;
        }
        return ST_ONLINE;  // Other possibilities???
    }


    /**
     *  Gets the output voltage
     *
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double getOutputVoltage() throws DriverException
    {
        String[] reply = receiveString("detstatus -om");
        checkReplyLength(reply, 9);
        return getDoubleValue(reply[0]);
    }


    /**
     *  Gets the output current
     *
     *  @return  The current
     *  @throws  DriverException
     */
    public double getOutputCurrent() throws DriverException
    {
        String[] reply = receiveString("detstatus -om");
        checkReplyLength(reply, 9);
        return getDoubleValue(reply[4]);
    }


    /**
     *  Gets the output power percent
     *
     *  @return  The power percent
     *  @throws  DriverException
     */
    public double getOutputPower() throws DriverException
    {
        String[] reply = receiveString("detstatus -om");
        checkReplyLength(reply, 9);
        return getDoubleValue(reply[2]);
    }


    /**
     *  Gets the battery temperature
     *
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    public double getTemperature() throws DriverException
    {
        String[] reply = receiveString("detstatus -tmp");
        checkReplyLength(reply, 3);
        return getDoubleValue(reply[0]);
    }


    /**
     *  Gets the battery voltage
     *
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double getBatteryVoltage() throws DriverException
    {
        String[] reply = receiveString("detstatus -bat");
        checkReplyLength(reply, 3);
        return getDoubleValue(reply[0]);
    }


    /**
     *  Gets the battery charge state
     *
     *  @return  The charge state (%)
     *  @throws  DriverException
     */
    public double getBatteryCharge() throws DriverException
    {
        String[] reply = receiveString("detstatus -soc");
        checkReplyLength(reply, 3);
        return getDoubleValue(reply[0]);
    }


    /**
     *  Gets the remaining run time
     *
     *  @return  The remaining run time (sec)
     *  @throws  DriverException
     */
    public int getRunTime() throws DriverException
    {
        String[] reply = receiveString("detstatus -rt");
        checkReplyLength(reply, 3);
        String[] words = getStringValue(reply[0]).split(" ");
        try {
            return Integer.parseInt(words[4]) + 60 * Integer.parseInt(words[2]) + 3600 * Integer.parseInt(words[0]);
        }
        catch (IndexOutOfBoundsException | NumberFormatException e) {
            responseError(reply[0]);
            return 0;  // Will never happen
        }
    }


    /**
     *  Sends a command and receives a string array
     *
     *  @param  command  The command to send
     *  @return  The string array
     *  @throws  DriverException
     */
    public synchronized String[] receiveString(String command) throws DriverException
    {
        String[] resp = receive(command);
        if (resp.length < 2) {
            throw new DriverException("No response received");
        }
        if (resp[1].matches("E\\d\\d\\d:.*")) {
            if (resp[1].startsWith("E00")) {
                String[] reply = new String[resp.length - 2];
                System.arraycopy(resp, 2, reply, 0, reply.length);
                return reply;
            }
            throw new DriverException("Command error: " + resp[1]);
        }
        else {
            throw new DriverException("Unexpected response: " + resp[1]);
        }
    }


    /**
     *  Gets the version information
     *
     *  @return  The saved prompt
     */
    private void getVersionInfo() throws DriverException
    {
        String[] reply = receiveString("upsabout");
        checkReplyLength(reply, 11);
        product = getStringValue(reply[0]);
        fwVersion = getStringValue(reply[2]);
    }


    /**
     *  Gets the string value from a line.
     * 
     *  Information lines all have the format desc: value 
     *
     *  @return  The string value
     *  @throws  DriverException
     */
    private String getStringValue(String line) throws DriverException
    {
        try {
            return line.split(": ")[1];
        }
        catch (IndexOutOfBoundsException e) {
            responseError(line);
            return null;  // Will never happen
        }
    }


    /**
     *  Gets the double value from a line.
     * 
     *  Numeric information lines all have the format desc: value units
     *
     *  @return  The string value
     *  @throws  DriverException
     */
    private static double getDoubleValue(String line) throws DriverException
    {
        try {
            return Double.parseDouble(line.split(": ")[1].split(" ")[0]);
        }
        catch (IndexOutOfBoundsException | NumberFormatException e) {
            responseError(line);
            return 0;  // Will never happen
        }
    }

}
