package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 ******************************************************
 * 
 *  Program to test the Agilent WG33250 device driver
 * 
 *  @author Owen Saxton
 * 
 ******************************************************
 */
public class TestWG33250 extends TestScpi {

    private final WG33250 wg;

    public enum OnOff {
        OFF, ON;
    }


   /**
    *  Constructor
    */
    public TestWG33250()
    {
        super(new WG33250());
        wg = (WG33250)scpi;
    }


   /**
    *  Opens connection to a device.
    * 
    *  @param  connType  The type of connection: ftdi or serial
    *
    *  @param  ident     The identification: serial no. (ftdi) or port name
    *                    (serial)
    *
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     WG33250.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        wg.open(connType, ident);
    }


   /**
    *  Starts the generator using specified parameters.
    *
    *  @param  func  The function to use
    *
    *  @param  freq  The frequency
    *
    *  @param  ampl  The amplitude
    *
    *  @param  offs  The offset
    *
    *  @throws  DriverException
    */
    @Command(name="start", description="Start the generator")
    public void start(@Argument(name="function", description="Function name")
                      WG33250.Function func,
                      @Argument(name="frequency", description="Frequency (Hz)")
                      double freq,
                      @Argument(name="amplitude", description="Amplitude (V)")
                      double ampl,
                      @Argument(name="offset", description="Offset (V)")
                      double offs) throws DriverException
    {
        wg.start(func, freq, ampl, offs);
    }


   /**
    *  Sets the generator function.
    *
    *  @param  func  The enumerated function
    *
    *  @throws  DriverException
    */
    @Command(name="setfunction", description="Set the function")
    public void setFunction(@Argument(name="function",
                                      description="Function name")
                            WG33250.Function func) throws DriverException
    {
        wg.setFunction(func);
    }


   /**
    *  Shows the function.
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showfunction", description="Show the function")
    public String getFunction() throws DriverException
    {
        return "Function = " + wg.getFunction().name().toLowerCase();
    }


   /**
    *  Sets the frequency.
    *
    *  @param  freq  The frequency
    *
    *  @throws  DriverException
    */
    @Command(name="setfrequency", description="Set the frequency")
    public void setFrequency(@Argument(name="frequency",
                                       description="Frequency (Hz)")
                             double freq) throws DriverException
    {
        wg.setFrequency(freq);
    }


   /**
    *  Shows the frequency.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showfrequency", description="Show the frequency")
    public String getFrequency() throws DriverException
    {
        return "Frequency = " + wg.getFrequency();
    }


   /**
    *  Shows a frequency limit.
    *
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showfrequency", description="Show a frequency limit")
    public String getFrequency(@Argument(name="limit", description="Limit name")
                               WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " frequency = " + wg.getFrequency(limit);
    }


   /**
    *  Sets the amplitude.
    *
    *  @param  ampl  The amplitude
    *
    *  @throws  DriverException
    */
    @Command(name="setamplitude", description="Set the amplitude")
    public void setAmplitude(@Argument(name="amplitude",
                                       description="Amplitude (V)")
                             double ampl) throws DriverException
    {
        wg.setAmplitude(ampl);
    }


   /**
    *  Shows the amplitude.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showamplitude", description="Show the amplitude")
    public String getAmplitude() throws DriverException
    {
        return "Amplitude = " + wg.getAmplitude();
    }


   /**
    *  Shows an amplitude limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showamplitude", description="Show a amplitude limit")
    public String getAmplitude(@Argument(name="limit", description="Limit name")
                               WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " amplitude = " + wg.getAmplitude(limit);
    }


   /**
    *  Sets the offset.
    *
    *  @param  offs  The offset
    *
    *  @throws  DriverException
    */
    @Command(name="setoffset", description="Set the offset")
    public void setOffset(@Argument(name="offset", description="Offset (V)")
                          double offs) throws DriverException
    {
        wg.setOffset(offs);
    }


   /**
    *  Shows the offset.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showoffset", description="Show the offset")
    public String getOffset() throws DriverException
    {
        return "Offset = " + wg.getOffset();
    }


   /**
    *  Shows an offset limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showoffset", description="Show a offset limit")
    public String getOffset(@Argument(name="limit", description="Limit name")
                            WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " offset = " + wg.getOffset(limit);
    }


   /**
    *  Sets the high level.
    *
    *  @param  level  The level
    *
    *  @throws  DriverException
    */
    @Command(name="sethighlevel", description="Set the high level")
    public void setHighLevel(@Argument(name="highlevel",
                                       description="High level (V)")
                             double level) throws DriverException
    {
        wg.setHighLevel(level);
    }


   /**
    *  Shows the high level.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showhighlevel", description="Show the high level")
    public String getHighLevel() throws DriverException
    {
        return "High level = " + wg.getHighLevel();
    }


   /**
    *  Shows a high level limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showhighlevel", description="Show a high level limit")
    public String getHighlevel(@Argument(name="limit",
                                         description="Limit name")
                               WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " high level = " + wg.getHighLevel(limit);
    }


   /**
    *  Sets the low level.
    * 
    *  @param  level  The level
    *
    *  @throws  DriverException
    */
    @Command(name="setlowlevel", description="Set the low level")
    public void setLowLevel(@Argument(name="lowlevel",
                                      description="Low level (V)")
                            double level) throws DriverException
    {
        wg.setLowLevel(level);
    }


   /**
    *  Shows the low level.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showlowlevel", description="Show the low level")
    public String getLowLevel() throws DriverException
    {
        return "Low level = " + wg.getLowLevel();
    }


   /**
    *  Shows a low level limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showlowlevel", description="Show a low level limit")
    public String getLowlevel(@Argument(name="limit", description="Limit name")
                              WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " low level = " + wg.getLowLevel(limit);
    }


   /**
    *  Sets the autorange state.
    *
    *  @param  state  The enumerated state
    *
    *  @throws  DriverException
    */
    @Command(name="setautorange", description="Set autorange state")
    public void setOutput(@Argument(name="state",
                                    description="Autorange state name")
                          WG33250.Autorange state) throws DriverException
    {
        wg.setAutorange(state);
    }


   /**
    *  Shows the autorange state.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showautorange", description="Show the autorange state")
    public String showAutorange() throws DriverException
    {
        return "Autorange = " + getOnOff(wg.isAutorangeOn());
    }


   /**
    *  Sets the amplitude unit.
    *
    *  @param  unit  The enumerated amplitude unit
    *
    *  @throws  DriverException
    */
    @Command(name="setunit", description="Set the amplitude unit")
    public void setUnit(@Argument(name="unit", description="Unit name")
                        WG33250.Unit unit) throws DriverException
    {
        wg.setUnit(unit);
    }


   /**
    *  Shows the amplitude unit.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showunit", description="Show the amplitude unit")
    public String getUnit() throws DriverException
    {
        return "Unit = " + wg.getUnit().name().toLowerCase();
    }


   /**
    *  Sets the square wave duty cycle.
    *
    *  @param  cycle  The duty cycle
    *
    *  @throws  DriverException
    */
    @Command(name="setdutycycle", description="Set the square wave duty cycle")
    public void setDutyCycle(@Argument(name="dutycycle",
                                       description="Duty cycle (%)")
                             double cycle) throws DriverException
    {
        wg.setDutyCycle(cycle);
    }


   /**
    *  Shows the square wave duty cycle.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showdutycycle", description="Show the square wave duty cycle")
    public String getDutyCycle() throws DriverException
    {
        return "Duty cycle = " + wg.getDutyCycle();
    }


   /**
    *  Shows a square wave duty cycle limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showdutycycle", description="Show a square wave duty cycle limit")
    public String getDutyCycle(@Argument(name="limit", description="Limit name")
                               WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " duty cycle = " + wg.getDutyCycle(limit);
    }


   /**
    *  Sets the ramp symmetry.
    *
    *  @param  symm  The ramp symmetry
    *
    *  @throws  DriverException
    */
    @Command(name="setsymmetry", description="Set the ramp symmetry")
    public void setSymmetry(@Argument(name="symmetry",
                                      description="Symmetry value (%)")
                            double symm) throws DriverException
    {
        wg.setSymmetry(symm);
    }


   /**
    *  Shows the ramp symmetry.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showsymmetry", description="Show the ramp symmetry")
    public String getSymmetry() throws DriverException
    {
        return "Symmetry = " + wg.getSymmetry();
    }


   /**
    *  Shows a ramp symmetry limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showsymmetry", description="Show a ramp symmetry limit")
    public String getSymmetry(@Argument(name="limit", description="Limit name")
                              WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " symmetry = " + wg.getSymmetry(limit);
    }


   /**
    *  Turns the output on or off.
    *
    *  @param  state  The state to set (ON or OFF)
    *
    *  @throws  DriverException
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state")
                          OnOff state) throws DriverException
    {
        wg.setOutput(state == OnOff.ON);
    }


   /**
    *  Shows the output state.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(wg.isOutputOn());
    }


   /**
    *  Sets the output load.
    *
    *  @param  load  The load
    *
    *  @throws  DriverException
    */
    @Command(name="setload", description="Set the output load")
    public void setOutputLoad(@Argument(name="load",
                                        description="Output load (ohms)")
                              double load) throws DriverException
    {
        wg.setOutputLoad(load);
    }


   /**
    *  Shows the output load.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showload", description="Show the output load")
    public String getOutputLoad() throws DriverException
    {
        return "Output load = " + wg.getOutputLoad();
    }


   /**
    *  Shows an output load limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showload", description="Show a output load limit")
    public String getOutputLoad(@Argument(name="limit", description="Limit name")
                                WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " output load = "
                 + wg.getOutputLoad(limit);
    }


   /**
    *  Sets the polarity.
    *
    *  @param  value  The enumerated polarity
    *
    *  @throws  DriverException
    */
    @Command(name="setpolarity", description="Set the polarity")
    public void setPolarity(@Argument(name="polarity",
                                      description="Polarity name")
                            WG33250.Polarity value) throws DriverException
    {
        wg.setPolarity(value);
    }


   /**
    *  Shows the polarity.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showpolarity", description="Show the polarity")
    public String showPolarity() throws DriverException
    {
        return "Polarity = " + wg.getPolarity().name().toLowerCase();
    }


   /**
    *  Turns the sync output on or off.
    *
    *  @param  state  The state to set (ON or OFF)
    *
    *  @throws  DriverException
    */
    @Command(name="setsync", description="Turn sync output on or off")
    public void setSyncOutput(@Argument(name="state",
                                        description="Sync output state")
                              OnOff state) throws DriverException
    {
        wg.setSyncOutput(state == OnOff.ON);
    }


   /**
    *  Shows the sync output state.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showsync", description="Show the sync output state")
    public String showSyncOutput() throws DriverException
    {
        return "Sync output state = " + getOnOff(wg.isSyncOutputOn());
    }


   /**
    *  Sets the pulse period.
    *
    *  @param  period  The period
    *
    *  @throws  DriverException
    */
    @Command(name="setperiod", description="Set the pulse period")
    public void setPulsePeriod(@Argument(name="period",
                                         description="Pulse period")
                               double period) throws DriverException
    {
        wg.setPulsePeriod(period);
    }


   /**
    *  Shows the pulse period.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showperiod", description="Show the pulse period")
    public String getPulsePeriod() throws DriverException
    {
        return "Pulse period = " + wg.getPulsePeriod();
    }


   /**
    *  Shows a pulse period limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showperiod", description="Show a pulse period limit")
    public String getPulsePeriod(@Argument(name="limit",
                                           description="Limit name")
                                 WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " pulse period = "
                 + wg.getPulsePeriod(limit);
    }


   /**
    *  Sets the pulse width.
    *
    *  @param  width  The pulse width
    *
    *  @throws  DriverException
    */
    @Command(name="setwidth", description="Set the pulse width")
    public void setPulseWidth(@Argument(name="width", description="Pulse width")
                              double width) throws DriverException
    {
        wg.setPulseWidth(width);
    }


   /**
    *  Shows the pulse width.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showwidth", description="Show the pulse width")
    public String getPulseWidth() throws DriverException
    {
        return "Pulse width = " + wg.getPulseWidth();
    }


   /**
    *  Shows a pulse width limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showwidth", description="Show a pulse width limit")
    public String getPulseWidth(@Argument(name="limit",
                                          description="Limit name")
                                WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " pulse width = "
                 + wg.getPulseWidth(limit);
    }


   /**
    *  Sets the pulse transition time.
    *
    *  @param  trans  The transition time
    *
    *  @throws  DriverException
    */
    @Command(name="settransition", description="Set the pulse transition time")
    public void setPulseTransition(@Argument(name="transition",
                                             description="Pulse transition")
                               double trans) throws DriverException
    {
        wg.setPulseTransition(trans);
    }


   /**
    *  Shows the pulse transition time.
    * 
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showtransition", description="Show the pulse transition time")
    public String getPulseTransition() throws DriverException
    {
        return "Pulse transition = " + wg.getPulseTransition();
    }


   /**
    *  Shows a pulse transition time limit.
    * 
    *  @param  limit  The enumerated limit to show
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="showtransition", description="Show a pulse transition time limit")
    public String getPulseTransition(@Argument(name="limit",
                                               description="Limit name")
                                     WG33250.Limit limit) throws DriverException
    {
        return capitalize(limit.name()) + " pulse transition = "
                 + wg.getPulseTransition(limit);
    }


   /**
    *  Capitalizes a string.
    */
    private static String capitalize(String str)
    {
        return str.substring(0, 1).toUpperCase() + str.substring(1).toLowerCase();
    }


   /**
    *  Converts a boolean to on/off.
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
