package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *****************************************************
 * 
 *  Program to test the Agilent DL34970 device driver
 * 
 *  @author Owen Saxton
 * 
 *****************************************************
 */
public class TestDL34970 extends TestScpi {

    private final DL34970 dl;


   /**
    *  Constructor
    */
    public TestDL34970()
    {
        super(new DL34970());
        dl = (DL34970)scpi;
    }


   /**
    *  Opens connection to a device.
    * 
    *  @param  connType  The type of connection: ftdi or serial
    *
    *  @param  ident     The identification: serial no. (ftdi) or port name
    *                    (serial)
    *
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     DL34970.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        dl.open(connType, ident);
    }


   /**
    *  Configures for reading a thermocouple.
    * 
    *  @param  type   The TC type
    *
    *  @param  chans  The channels to configure
    *
    *  @throws  DriverException
    */
    @Command(name="configtc", description="Configure for a thermocouple")
    public void configTC(@Argument(name="type", description="TC type")
                         DL34970.TcType type,
                         @Argument(name="chans", description="Channel numbers")
                         int... chans) throws DriverException
    {
        dl.configTC(type, chans);
    }


   /**
    *  Configures for reading a voltage.
    * 
    *  @param  range  The voltage range
    *
    *  @param  chans  The channels to configure
    *
    *  @throws  DriverException
    */
    @Command(name="configvolts", description="Configure for a voltage")
    public void configVolts(@Argument(name="range", description="Voltage range")
                            double range,
                            @Argument(name="chans", description="Channel numbers")
                            int... chans) throws DriverException
    {
        dl.configVolts(range, chans);
    }


   /**
    *  Configures for reading a current.
    * 
    *  @param  range  The current range
    *
    *  @param  chans  The channels to configure
    *
    *  @throws  DriverException
    */
    @Command(name="configcurrent", description="Configure for a current")
    public void configCurrent(@Argument(name="range", description="Current range")
                              double range,
                              @Argument(name="chans", description="Channel numbers")
                              int... chans) throws DriverException
    {
        dl.configCurrent(range, chans);
    }


   /**
    *  Sets the scan list.
    * 
    *  @param  chans  The channels to set in the scan list
    *
    *  @throws  DriverException
    */
    @Command(name="setscanlist", description="Set the scan list")
    public void setScanList(@Argument(name="chans", description="Channel numbers")
                            int... chans) throws DriverException
    {
        dl.setScanList(chans);
    }


   /**
    *  Reads channels.
    *
    *  @param  chans  The channel numbers.  If empty, the current scan list
    *                 is used; otherwise the scan list is set to contain
    *                 these channels.
    *
    *  @return  The result string
    *
    *  @throws  DriverException
    */
    @Command(name="readdata", description="Reads data from channels")
    public String readData(@Argument(name="chans", description="Channel numbers")
                           int... chans) throws DriverException
    {
        if (chans.length == 1) {
            return "Value = " + dl.readData(chans[0]);
        }
        double[] data = chans.length == 0 ? dl.readData() : dl.readData(chans);
        StringBuilder text = new StringBuilder("Values =");
        for (double value : data) {
            text.append(' ').append(value);
        }
        return text.toString();
    }

}
