package org.lsst.ccs.drivers.agilent;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 ******************************************************************************
 **
 **  Routines for controlling an Agilent N6700 series modular power supply
 **
 **  @author  Owen Saxton
 **
 ******************************************************************************
 */
public class N6700 extends Scpi {

    public static final int
        VOLTAGE_MODE     = 0,
        CURRENT_MODE     = 1,
        CHAN_INFO_MODEL  = 0,
        CHAN_INFO_OPTION = 1,
        CHAN_INFO_SERIAL = 2,
        MIN_CHANNEL      = 1,
        MAX_CHANNEL      = 4,
        DEFAULT_PORT     = 5025;

    private int mode = VOLTAGE_MODE;


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String hostname) throws DriverException
    {
        open(hostname, DEFAULT_PORT);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    ***************************************************************************
    */
    public void open(String hostname, int port) throws DriverException
    {
        open(Scpi.CONN_TYPE_NETWORK, hostname, port);
        checkIdentification("Agilent", Scpi.CHECK_STARTS_WITH,
                            "N67", Scpi.CHECK_STARTS_WITH);
    }


   /**
    ***************************************************************************
    **
    **  Turns output on or off.
    **
    ***************************************************************************
    */
    public void setOutput(boolean on, int chan) throws DriverException
    {
        writeAg("OUTP " + (on ? "ON" : "OFF"), chan);
    }


   /**
    ***************************************************************************
    **
    **  Turns output on or off.
    **
    ***************************************************************************
    */
    public void setOutput(boolean on, int[] chan) throws DriverException
    {
        writeAg("OUTP " + (on ? "ON" : "OFF"), chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    ***************************************************************************
    */
    public boolean getOutput(int chan) throws DriverException
    {
        return readIntegerAg("OUTP?", chan) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Gets the output state.
    **
    ***************************************************************************
    */
    public boolean[] getOutput(int[] chan) throws DriverException
    {
        int[] iState = readIntegerAg("OUTP?", chan);
        boolean[] state = new boolean[iState.length];
        for (int j = 0; j < state.length; j++) {
            state[j] = iState[j] != 0;
        }

        return state;
    }


   /**
    ***************************************************************************
    **
    **  Sets the output-on delay.
    **
    ***************************************************************************
    */
    public void setOnDelay(double time, int chan) throws DriverException
    {
        writeAg("OUTP:DEL:RISE " + time, chan);
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the output-on delay.
    **
    ***************************************************************************
    */
    public double getOnDelay(int chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:RISE?", chan);
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the output-on delay.
    **
    ***************************************************************************
    */
    public double[] getOnDelay(int[] chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:RISE?", chan);
    }
    

   /**
    ***************************************************************************
    **
    **  Sets the output-off delay.
    **
    ***************************************************************************
    */
    public void setOffDelay(double time, int chan) throws DriverException
    {
        writeAg("OUTP:DEL:FALL " + time, chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the output-off delay.
    **
    ***************************************************************************
    */
    public double getOffDelay(int chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:FALL?", chan);
    }
    

   /**
    ***************************************************************************
    **
    **  Gets the output-off delay.
    **
    ***************************************************************************
    */
    public double[] getOffDelay(int[] chan) throws DriverException
    {
        return readDoubleAg("OUTP:DEL:FALL?", chan);
    }
    

   /**
    ***************************************************************************
    **
    **  Sets the voltage.
    **
    ***************************************************************************
    */
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeAg("VOLT " + value, chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set voltage.
    **
    ***************************************************************************
    */
    public double getVoltage(int chan) throws DriverException
    {
        return readDoubleAg("VOLT?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set voltage.
    **
    ***************************************************************************
    */
    public double[] getVoltage(int[] chan) throws DriverException
    {
        return readDoubleAg("VOLT?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    ***************************************************************************
    */
    public double readVoltage(int chan) throws DriverException
    {
        return readDoubleAg("MEAS:VOLT?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Reads the voltage.
    **
    ***************************************************************************
    */
    public double[] readVoltage(int[] chan) throws DriverException
    {
        return readDoubleAg("MEAS:VOLT?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current.
    **
    ***************************************************************************
    */
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeAg("CURR " + value, chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current.
    **
    ***************************************************************************
    */
    public double getCurrent(int chan) throws DriverException
    {
        return readDoubleAg("CURR?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the set current.
    **
    ***************************************************************************
    */
    public double[] getCurrent(int[] chan) throws DriverException
    {
        return readDoubleAg("CURR?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    ***************************************************************************
    */
    public double readCurrent(int chan) throws DriverException
    {
        return readDoubleAg("MEAS:CURR?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Reads the current.
    **
    ***************************************************************************
    */
    public double[] readCurrent(int[] chan) throws DriverException
    {
        return readDoubleAg("MEAS:CURR?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets channel information.
    **
    ***************************************************************************
    */
    public String[] getChannelInfo(int chan) throws DriverException
    {
        String[] data = new String[3];
        data[CHAN_INFO_MODEL]  = readStringAg("SYST:CHAN:MODEL?", chan);
        data[CHAN_INFO_OPTION] = readStringAg("SYST:CHAN:OPTION?", chan);
        data[CHAN_INFO_SERIAL] = readStringAg("SYST:CHAN:SERIAL?", chan);

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Sets the priority mode (voltage or current).
    **
    ***************************************************************************
    */
    public void setPriorityMode(int vctype, int chan) throws DriverException
    {
        writeAg("FUNC " + (vctype == VOLTAGE_MODE ? "VOLT" : "CURR"), chan);
        mode = vctype;
    }


   /**
    ***************************************************************************
    **
    **  Sets the slew rate.
    **
    ***************************************************************************
    */
    public void setSlewRate(double rate, int chan) throws DriverException
    {
        writeAg((mode == VOLTAGE_MODE ? "VOLT" : "CURR") + ":SLEW " + rate,
                chan);
    }


   /**
    ***************************************************************************
    **
    **  Sets the voltage limit.
    **
    ***************************************************************************
    */
    public void setVoltageLimit(double maxima, int chan) throws DriverException
    {
        writeAg("VOLT:LIM " + maxima, chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the voltage limit.
    **
    ***************************************************************************
    */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDoubleAg("VOLT:LIM?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Sets the current limit.
    **
    ***************************************************************************
    */
    public void setCurrentLimit(double maxima, int chan) throws DriverException
    {
        writeAg("CURR:LIM " + maxima, chan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current limit.
    **
    ***************************************************************************
    */
    public double getCurrentLimit(int chan) throws DriverException
    {
        return readDoubleAg("CURR:LIM?", chan);
    }


   /**
    ***************************************************************************
    **
    **  Writes a command.
    **
    ***************************************************************************
    */
    private void writeAg(String instr, int chan) throws DriverException
    {
        writeCommand(instr + "," + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command.
    **
    ***************************************************************************
    */
    private void writeAg(String instr, int[] chan) throws DriverException
    {
        if (chan.length == 0) return;
        writeCommand(instr + "," + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the (string) result.
    **
    ***************************************************************************
    */
    private String readStringAg(String instr, int chan) throws DriverException
    {
        return readString(instr + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the (string) result.
    **
    ***************************************************************************
    */
    private String[] readStringAg(String instr, int[] chan)
        throws DriverException
    {
        if (chan.length == 0) return new String[0];
        return readStringArray(instr + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the double result.
    **
    ***************************************************************************
    */
    private double readDoubleAg(String instr, int chan) throws DriverException
    {
        return readDouble(instr + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the double result.
    **
    ***************************************************************************
    */
    private double[] readDoubleAg(String instr, int[] chan)
        throws DriverException
    {
        if (chan.length == 0) return new double[0];
        return readDoubleArray(instr + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the integer result.
    **
    ***************************************************************************
    */
    private int readIntegerAg(String instr, int chan) throws DriverException
    {
        return readInteger(instr + makeChannelList(chan));
    }


   /**
    ***************************************************************************
    **
    **  Writes a command and reads the integer result.
    **
    ***************************************************************************
    */
    private int[] readIntegerAg(String instr, int[] chan) throws DriverException
    {
        if (chan.length == 0) return new int[0];
        return readIntegerArray(instr + makeChannelList(chan));
    }
        
}
