package org.lsst.ccs.drivers.ads;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Contains all Instances of VariableHandle in use by the ADS driver.
 * Any instance may be found at any time by using its variable name.
 * If notification is in effect for the VariableHandle then it may be found
 * using the notification handle and that handle may be found using
 * the VariableHandle instance. Removing a VariableHandle will
 * remove the associations between it and its name and any notification
 * handle.
 * @author tether
 */
public class VarHandleTracker {
    
    private Set<VariableHandle> all;
    
    private Map<Integer, VariableHandle> fromNoticeHandle;
    
    private Map<String, VariableHandle>  fromName;
    
    private Map<VariableHandle, Integer> toNoticeHandle;

    /**
     * The set of VariableHandles is initially empty.
     */
    public VarHandleTracker() {
        this.all = new HashSet<>();
        this.fromNoticeHandle = new HashMap<>();
        this.fromName = new HashMap<>();
        this.toNoticeHandle = new HashMap<>();
    }
    
    public void add(final VariableHandle newvh) {
        if (all.contains(newvh)) return;
        all.add(newvh);
        fromName.put(newvh.getName(), newvh);
    }
    
    public boolean isValid(final VariableHandle vh) {
        return all.contains(vh);
    }
    
    public void remove(final VariableHandle vh) {
        all.remove(vh);
        fromName.remove(vh.getName());
        removeNotification(vh);
    }
    
    public void clear() {
        all.clear();
        fromName.clear();
        fromNoticeHandle.clear();
        toNoticeHandle.clear();
    }
    public VariableHandle getByName(final String name) {
        return fromName.get(name);
    }
    
    public void addNotification(final VariableHandle vh, int nh) {
        fromNoticeHandle.put(nh, vh);
        toNoticeHandle.put(vh, nh);
    }
    
    public Integer getNotification(final VariableHandle vh) {
        return toNoticeHandle.get(vh);
    }
    
    public VariableHandle getByNotification(final Integer nh) {
        return fromNoticeHandle.get(nh);
    }
    
    public Integer removeNotification(final VariableHandle vh) {
        final Integer nh = toNoticeHandle.remove(vh);
        if (nh != null) {
            fromNoticeHandle.remove(nh);
        }
        return nh;
    }
}
