package org.lsst.ccs.daq.utilities;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.Agent;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * A set of commands to publish HeaderKeyword data.
 * 
 * @author The LSST CCS Team
 */
public class FitsServiceHeaderCommands {
    
    private final Agent agent;
    
    public FitsServiceHeaderCommands(Agent agent) {
        this.agent = agent;
    }
    
    /**
     *  Set the value of a Header Keyword to be shared by all Fits extensions. 
     *
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.ACTION, description="Set a header keyword value shared by all extensions")
    public void setHeaderKeyword(String name, Serializable value) {
        Map<String,Serializable> map = new HashMap<>();
        map.put(name, value);
        setNamedHeaderKeywords(FitsService.COMMON_HEADER_NAME, map);
    }

    /**
     *  Set the value of a set of Header Keywords to be shared by all Fits extensions. 
     *  @param  headersMap The map of header keyword name, value pairs to be set
     *
     */
    @Command(type=Command.CommandType.ACTION, description="Set a header keyword value shared by all extensions")
    public void setHeaderKeywords(Map<String, Serializable> headersMap) {
        setNamedHeaderKeywords(FitsService.COMMON_HEADER_NAME, headersMap);
    }
    
    /**
     *  Set the value of a Primary extension Header Keyword
     *
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.ACTION, description="Set a primary header keyword value")
    public void setPrimaryHeaderKeyword(String name, Serializable value) {
        Map<String,Serializable> map = new HashMap<>();
        map.put(name, value);
        setNamedHeaderKeywords("primary", map);
    }

    /**
     *  Set the values of Primary Header Keywords
     *
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.ACTION, description="Set primary header keyword values")
    public void setPrimaryHeaderKeywords(Map<String, Serializable> headersMap) {
        setNamedHeaderKeywords("primary", headersMap);
    }
   

    /**
     *  Set the value of a header keyword for a given named HDU.
     *
     *  @param  ccdId The Id of the CCD for which this keyword is published
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     *  @param  sticky If this value is sticky
     */
    @Command(type=Command.CommandType.ACTION, description="Set a header keyword value for a given CCD extension")
    public void setHeaderKeywordForCCD(String ccdId, String name, Serializable value, @Argument(defaultValue = "false") boolean sticky) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData(ccdId);
        data.addHeaderKeywordValue(ccdId,name, value, sticky);
        agent.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

    /**
     *  Set the values of a named HDU Header Keywords
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.ACTION, description="Set header keyword values on a named HDU")
    public void setNamedHeaderKeywords(String hduName, Map<String, Serializable> headersMap) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData();
        for ( Map.Entry<String,Serializable> e : headersMap.entrySet() ) {
            data.addHeaderKeywordValue(hduName, e.getKey(), e.getValue(), false);
        }
        agent.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

}
