package org.lsst.ccs.daq.utilities;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.bus.data.KeyValueData;

/**
 * Class containing data for setting header keywords.
 * This class is meant to be sent on the buses and intercepted by fits file
 * writing clients.
 * 
 * Since there might be many fits file writing clients these messages must find
 * their target. To do so we add an id corresponding to the hardware that is
 * publishing the data.
 * 
 * @author The LSST CCS Team
 */
@SkipEncoding
public class FitsHeaderKeywordData implements Serializable {
    
    private static final long serialVersionUID = 943756459784235235L;
    
    public final static String DATA_KEY = "fitsHeaderKeywordData";
    
    private final String dataId;
    private final List<HeaderKeywordValue> keywordValues = new ArrayList<>();
    
    public FitsHeaderKeywordData() {
        this(null);
    }

    public FitsHeaderKeywordData(String dataId) {
        this.dataId = dataId;
    }
    
    public void addHeaderKeywordValue(String headerName, String headerKeywordName, Serializable headerKeywordValue) {
        addHeaderKeywordValue(headerName, headerKeywordName, headerKeywordValue, false);
    }
    public void addHeaderKeywordValue(String headerName, String headerKeywordName, Serializable headerKeywordValue, boolean sticky) {
        keywordValues.add(new HeaderKeywordValue(headerName, headerKeywordName, headerKeywordValue, sticky));        
    }
    
    public List<HeaderKeywordValue> getHeaderKeywordValues() {
        return new ArrayList<>(keywordValues);
    }
    
    public String getDataId() {
        return dataId;
    }
    
    //Private class used to set Header Keyword values.
    public static class HeaderKeywordValue implements Serializable {

        private final String headerName;
        private final String headerKeywordName;
        private final Serializable headerKeywordValue;
        private final boolean sticky;

        HeaderKeywordValue(String headerName, String headerKeywordName, Serializable headerKeywordValue) {
            this(headerName, headerKeywordName, headerKeywordValue, false);
        }
        HeaderKeywordValue(String headerName, String headerKeywordName, Serializable headerKeywordValue, boolean sticky) {
            this.headerName = headerName;
            this.headerKeywordName = headerKeywordName;
            this.headerKeywordValue = headerKeywordValue;
            this.sticky = sticky;
        }

        public String getHeaderName() {
            return headerName;
        }

        public String getHeaderKeywordName() {
            return headerKeywordName;
        }

        public Serializable getHeaderKeywordValue() {
            return headerKeywordValue;
        }

        public boolean isSticky() {
            return sticky;
        }

    }
    
    public KeyValueData getKeyValueData() {
        return new KeyValueData(DATA_KEY,this);
    }
    
}
