package org.lsst.ccs.config;

/**
 * DAO interface to configuration data.
 * This interface is to be implemented by classes that can save/load configuration
 * data for a SingleCategoryTag.
 * 
 * @author LSST CCS Team
 */
public interface ConfigurationDAO extends AutoCloseable {
    
    /**
     * Load the configuration data for a SingleCategoryTag.
     * The provided SingleCategoryTag could contain generic versions as: default,
     * latest, working. It is the responsibility of the DAO to resolve this
     * version and return it as part of the SingleCategoryTag object it returns.
     * 
     * @param singleCategoryTag The object describing the data to be loaded for
     *                          a single category tag.
     * @return a SingleCategoryTagData containing the loaded configuration values
     *         and the comments associated to them. This object contains a 
     *         SingleCategoryTag instance to clarify the version of the data that
     *         was actually loaded.
     * 
     * @throws org.lsst.ccs.config.ConfigurationServiceException if the data 
     *         cannot be found or loaded for the given SingleCategoryTag instance.
     * 
     */
    public SingleCategoryTagData loadCategoryTag(SingleCategoryTag singleCategoryTag) throws ConfigurationServiceException ;
    
    /**
     * Save the data included in the provided SingleCategoryTag object.
     * It is the responsibility of the DAO to resolve the version of the saved
     * data and return it as part of the SingleCategoryTag object it returns.
     * 
     * @param singleCategoryTagData The data to be saved; it includes the configuration
     *                              parameter values and associated comments.
     * @return a SingleCategoryTag instance containing the version in which the
     *         provided data was saved.
     * 
     * @throws ConfigurationServiceException if the data could not be saved.
     */
    public SingleCategoryTag saveCategoryTagData(SingleCategoryTagData singleCategoryTagData) throws ConfigurationServiceException ;
    
    public default String resolveVersionForCategoryTag(SingleCategoryTag singleCategoryTag) throws ConfigurationServiceException {
        return singleCategoryTag.getRequestedVersion();
    }

    /**
     * Called when the DAO is no longer used (typically at shutdown).
     */
    @Override
    public default void close() {
        /// No-op by default
    }   
}
