package org.lsst.ccs.config;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.logging.Logger; 
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A set of several ConfigurationHandlers.
 * @author LSST CCS Team
 */
public class ConfigurationHandlerSet {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    private final  List<DataProviderInfo> list = new ArrayList();
    private static final Pattern categoryPattern = Pattern.compile("^[a-zA-Z0-9]+$");    
    
    
    // The following fields are created and populated when the Agent is still
    // single threaded, so they are thread safe as long as their content is thread safe.    
    /** 
     * Configurable parameters ordered by the component they belong to. 
     * A LinkedHashMap is used to preserve insertion order, which is the order with which the 
     * configuration handlers have to be traveled when applying configurations / processing bulk changes
     */
    private final Map<String, ConfigurationHandler> componentParameterHandlers = new LinkedHashMap<>();

    /** 
     * Configurable parameters ordered by the category they belong to. 
     */
    private final Map<String, CategoryHandler> parametersByCategories = new HashMap<>();
    
    private ConfigurationView initialView;
    
    private final Set<String> categories = new HashSet<>();
    private final Map<String,String> componentToPath = new HashMap<>();
    
    private final boolean checkConfigView;
    private final boolean failOnViewProblems;
    private boolean failForBuildParameters = false;
    private boolean failForFinalParameters = false;
    
    public ConfigurationHandlerSet(boolean checkConfigView, boolean failOnViewProblems) {
        this.checkConfigView = checkConfigView;
        this.failOnViewProblems = failOnViewProblems;
    }
    
    public ConfigurationHandlerSet() {
        this(true, false);
    }
    
    // -- Construction phase protected methods 
    //This method is only invoked by the ConfigurationService during the HasLifecycle steps
    //when the agent is still single threaded.
    public ConfigurationHandler addConfigurationHandlerForObject(String componentName, String componentPath, Object component) { 
        ConfigurationHandler handler = ConfigurationHandlerBuilder.buildParameterSetterFromObject(componentName, component);
        componentToPath.put(componentName, componentPath);
        
        if (handler == null) return null;
        
        componentParameterHandlers.put(componentName, handler);
        for (ConfigurationParameterHandler cph : handler.getConfigurationParameterHandlers()) {
            //Assign "build" category to all build parameters
            if (cph.isBuild()) {
                if ( cph.getCategory() != null && ! cph.getCategory().equals("build") ) {
                    log.log(Level.WARNING, "Changing assigned category {0} for isBuild parameter {1} to \"build\".\n To get rid of this warning remove \"isBuild=true\" in your ConfigurationParameter annotation and specify \"category=\"build\"\" instead.", new Object[] {cph.getCategory(),cph.getParameterPath()});
                }
                cph.setCategory("build");
            }
            String category = cph.getCategory();
            if ( ! categoryPattern.matcher(category).matches() ) {
                throw new RuntimeException("Illegal category name: \""+category+"\" for parameter "+cph.getParameterPath()+". Category names can only contain letters or digits.");
            }
            
            CategoryHandler set = parametersByCategories.get(category);
            if (set == null) {
                categories.add(category);
                set = new CategoryHandler(category);
                parametersByCategories.put(category, set);
            }
            set.addParameterHandler(cph);
        }
        return handler;
    }
    
    public ConfigurationInfo initialize(String descriptionName, String agentName) {
        return initialize(descriptionName, new ConfigurationView(ConfigurationDescription.initialDefaultConfiguration(categories)), agentName);
    }
    
    public ConfigurationInfo initialize(String descriptionName, ConfigurationView safeView, String agentName) {
        return initialize(descriptionName, safeView, agentName, false, false);        
    }
    public ConfigurationInfo initialize(String descriptionName, ConfigurationView safeView, String agentName, boolean isBuild, boolean isInit) {        
        failForBuildParameters = !isBuild;
        failForFinalParameters = !isBuild && !isInit;
        initialView = new ConfigurationView(ConfigurationDescription.initialDefaultConfiguration(categories));
        initialView.putAll(getLiveConfigurationView());        
        //If we are in the initialization step we need to check if the configuration parameter fields
        //are declared as volatile and if the collections have maxLength defined.
        if ( isInit && ! isBuild ) {
            List<ParameterPath> missingMaxLength = new ArrayList<>();
            List<ParameterPath> missingVolatile = new ArrayList<>();
            for ( ConfigurationHandler handler : componentParameterHandlers.values() ) {
                for ( ConfigurationParameterHandler configHandler : handler.getConfigurationParameterHandlers() ) {
                    boolean mustBeVolatile = !configHandler.isBuild() && !configHandler.isFinal();
                    if ( mustBeVolatile && !isVolatile(configHandler) ) {
                        missingVolatile.add(configHandler.getParameterPath());                                                    
                    }
                    if ( configHandler.hasLength() && configHandler.getMaxLength() == 0 ) {
                        missingMaxLength.add(configHandler.getParameterPath());                                                    
                    }
                }
            }
            if ( missingMaxLength.size() > 0 ) {
                log.log(Level.WARNING,"Collections or Arrays annotated as Configuration Parameters must provide the \"maxLength\" argument.\n"
                        + "Please update your code to specify the maximum allowed length for the following parameters: \n{0}",missingMaxLength);
            }
            if ( missingVolatile.size() > 0 ) {
                log.log(Level.WARNING,"Fields annotated as Configuration Parameters must be declared as \"volatile\".\n"
                        + "Please update your code for the following parameters:\n{0}",missingVolatile);
            }
        }
        
        
        
        try {
            safeView = loadCategories(safeView, isBuild, isInit);                                
            //Updating the initialView with all the parameters set by the safeView.
            initialView.putAll(safeView);            
        } catch (Exception ex) {
            throw new RuntimeException("could not load initial configuration.", ex);
        }
        
        
        // First configurationInfo creation
        ConfigurationInfo.Builder ciBuilder = new ConfigurationInfo.Builder()
                .setDescription(descriptionName).updateConfigurationDescription(safeView.getConfigurationDescription().getFullDescriptionName(), safeView.getConfigurationDescription().getDescriptionName());
        for (CategoryHandler ch : parametersByCategories.values()) {
            String category = ch.getCategory();

            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                ParameterPath pp = new ParameterPath(cph.getComponentName(), cph.getParameterName());
                ciBuilder.addParameter(pp, cph.getType(), cph.getCategory(), cph.getDescription(), cph.isFinal(), cph.isReadOnly(), cph.isBuild());
                
                DataProviderInfo dataProviderInfo = new DataProviderInfo(componentToPath.get(pp.getComponentName()), DataProviderInfo.Type.CONFIGURATION, pp.getParameterName());
                if ( !cph.getUnits().isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.UNITS, cph.getUnits());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.TYPE, cph.getType().getTypeName());
                if ( !cph.getDescription().isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, cph.getDescription());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_TYPE, cph.getParameterType().name());
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_CATEGORY, cph.getCategory());
                String range = cph.getRange();
                if ( range != null && !range.isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_RANGE, range);
                }
                if ( cph.hasLength() && cph.getMaxLength() != 0 ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_MAX_LENGTH, String.valueOf(cph.getMaxLength()));                    
                }
                list.add(dataProviderInfo);
                
                ciBuilder.updateParameter(pp, initialView.getPathValue(pp), initialView.getPathValue(pp), false);
            }
        }
        Collections.sort(list,comparator);
        failForBuildParameters = true;
        failForFinalParameters = true;        
        return ciBuilder.setConfigurationState(ConfigurationState.UNCONFIGURED)
                .setCCSTimeStamp(CCSTimeStamp.currentTime()).build();
    }
    
    public ConfigurationParameterHandler getConfigurationParameterHandler(ParameterPath pp) {
        return componentParameterHandlers.get(pp.getComponentName()).getConfigurationParameterHandler(pp.getParameterName());
    }
    
    private static boolean isVolatile(ConfigurationParameterHandler parameterHandler) {
        return Modifier.isVolatile(parameterHandler.getField().getModifiers()) || Modifier.isFinal(parameterHandler.getField().getModifiers());        
    }

    private static Comparator<DataProviderInfo> comparator = new Comparator<DataProviderInfo>() {
        @Override
        public int compare(DataProviderInfo o1, DataProviderInfo o2) {
            return o1.getFullPath().compareTo(o2.getFullPath());
        }
        
    };

    public List<DataProviderInfo> getDataPoviderInfoList() {
        return list;
    }
    
    // -- End construction phase protected methods.
    
    public ConfigurationHandler getConfigurationHandlerForGroovy(String componentName) {
        try {
            return getParameterSet(componentName);
        } catch (Exception e) {
//            e.printStackTrace();
            return null;
        }
    }
    
    public ConfigurationHandler getParameterSet(String componentName) {
        ConfigurationHandler res = componentParameterHandlers.get(componentName);
        if (res == null) {
            throw new IllegalArgumentException("no such component : " + componentName);
        }
        return res;
    }

    public CategoryHandler getCategoryHandler(String category) {
        CategoryHandler res = parametersByCategories.get(category);
        if (res == null) {
            log.info("no such category : \"" + category+"\"");
            throw new RuntimeException("Category \""+category+"\" does not exist");
        }
        return res;
    }
    
    public boolean isBuildParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isBuildParameter(parameterName);
    }

    public boolean isFinalParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isFinalParameter(parameterName);
    }

    public boolean isParameterConfigurable(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterConfigurable(parameterName);
    }
    
    public boolean isParameterReadOnly(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterReadOnly(parameterName);
    }

    public Set<String> getCategorySet() {
        return Collections.unmodifiableSet(parametersByCategories.keySet());
    }
    
    public ConfigurationView loadCategories(ConfigurationView cv) {
        return loadCategories(cv,false, false);
    }
    
    private boolean isValueEqualToLoadedValue(String componentName, String parameterName, Object value) {
        ConfigurationParameterHandler cph = getParameterSet(componentName).getConfigurationParameterHandler(parameterName);
        Object inputValue = value instanceof String ? cph.convert((String)value).getValue() : value;
        Object currentValue = cph.getObjectValue();
        if ( inputValue == null ) {
            return currentValue == null;
        } else {
            return Objects.deepEquals(inputValue, currentValue);
        }
    }
        
    private ConfigurationView loadCategories(ConfigurationView cv, boolean isBuild, boolean isInit) {                
        ConfigurationDescription configDesc = cv.getConfigurationDescription();
        
        StringBuilder unexpectedParms = new StringBuilder("");
        for (Map.Entry<ParameterPath, String> entry : cv.getAsParameterPathMap().entrySet()) {
            ParameterPath parm = entry.getKey();
            //At build time handle only build configuration parameters
            String componentName = parm.getComponentName();
            String parameterName = parm.getParameterName();
            if ( isParameterReadOnly(componentName, parameterName) ) {
                continue;
            }
            if (isParameterConfigurable(componentName, parameterName) && ! isParameterReadOnly(componentName, parameterName)) {
                if ( isInit || ! isValueEqualToLoadedValue(componentName, parameterName, entry.getValue()) ) {
                    Object newValue = submitChange(componentName, parameterName, entry.getValue());
                    cv.putParameterValue(componentName, parameterName, (String)newValue);
                }
            } else {
                if ( cv.containsPath(parm) ) {
                    unexpectedParms.append(parm).append(" ");
                }
            }
        }
        
        // Submitting the initial value for other components
        // REVIEW: should we do this ?
        if ( configDesc != null ) {
            Map<String, String> missingParmsDigest = new HashMap<>();
            for (String category : configDesc.getCategoriesSet()) {
                StringBuilder missingParms = new StringBuilder("");
                CategoryHandler ch = getCategoryHandler(category);
                for (ConfigurationParameterHandler cph : ch.getParameters()) {
                    String componentName = cph.getComponentName();
                    String parameterName = cph.getParameterName();
                    ParameterPath parm = new ParameterPath(componentName, parameterName);

                    //The following parameters must be present when configurations are loaded:
                    //- non readonly and non final and non build at all times
                    //- final parameters during the init or build phase
                    //- build parameters during the build phase
                    if (!cv.containsPath(parm)) {
                        boolean isRuntime = (!cph.isReadOnly() && !cph.isFinal() && !cph.isBuild());
                        boolean isMissingFinal = (cph.isFinal() && (isInit || isBuild));
                        boolean isMissingBuild = (cph.isBuild() && isBuild);
                        if (isRuntime || isMissingFinal || isMissingBuild) {
                            missingParms.append(new ParameterPath(componentName, parameterName).toString()).append(" ");
                        }
                    }
                }
                if (!missingParms.toString().isEmpty()) {
                    missingParmsDigest.put("\"" + configDesc.getCategoryTag(category) + "\"", missingParms.toString());
                }
            }
            
            // Configuration content diagnostic
            if (checkConfigView) {
                if (!unexpectedParms.toString().isEmpty() || !missingParmsDigest.isEmpty()) {
                    StringBuilder diag = new StringBuilder("the following problems occured when reading the configuration : ").append(configDesc.toString()).append("\n");
                    if (!unexpectedParms.toString().isEmpty()) {
                        diag.append("\t unexpected parameters : ").append(unexpectedParms.toString()).append("\n");
                    }
                    if (!missingParmsDigest.isEmpty()) {
                        for (Map.Entry<String, String> e : missingParmsDigest.entrySet()) {
                            diag.append("\t missing values for configuration ").append(e.getKey()).append(" : ").append(e.getValue()).append("\n");
                        }
                    }
                    log.warn(diag.toString() + "Saving the problematic configurations once they are loaded will fix them.");
                                        
                    if (BootstrapResourceUtils.getBootstrapSystemProperties().getOrDefault(PackCst.REQUIRE_COMPLETE_CONFIGURATIONS, "false").equals("true") || (failOnViewProblems && !System.getProperty("org.lsst.ccs.testcontext", "false").equals("true"))) {
                        throw new RuntimeException(diag.append("Load aborted.").toString());
                    }
                }
            }
        }        

        return  commitBulkChange(configDesc,isInit, isBuild);
    }
        
    public void dropAllSubmittedChanges() {
        componentParameterHandlers.values()
                .stream().forEach(ConfigurationHandler::dropSubmittedChanges);
    }
    
    public void dropSubmittedChangesForComponent(String name) {
        getParameterSet(name).dropSubmittedChanges();
    }
    
    public Map<String, String> getSubmittedChangesForComponent(String name) {
        return getParameterSet(name).getSubmittedChanges().entrySet()
                .stream().collect(Collectors.toMap(
                        entry -> entry.getKey(), 
                        entry -> entry.getValue()));
    }
    
    public Map<String, Map<String, String>> getAllSubmittedChanges() {
        Map<String, Map<String, String>> res = new TreeMap<>();
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            Map<String, String> submittedChanges = getSubmittedChangesForComponent(entry.getKey());
            if (!submittedChanges.isEmpty()) {
                res.put(entry.getKey(), new TreeMap<String, String>(submittedChanges));
            }
        }
        return res;
    }
    
    public Object submitChange(String componentName, String parameterName, Object value, boolean readOnlyOk) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            throw new IllegalArgumentException(componentName+"/"+parameterName + " is not a valid configuration parameter.");
        } else if (isParameterReadOnly(componentName, parameterName) && !readOnlyOk) {
            throw new IllegalArgumentException(componentName+"/"+parameterName + " is a read-only configuration parameter.");
        } else if (isBuildParameter(componentName, parameterName) && failForBuildParameters) {
            if ( ! isValueEqualToLoadedValue(componentName, parameterName, value) ) {
                throw new IllegalArgumentException(componentName+"/"+parameterName + " is a build configuration parameter and cannot be modified at runtime");
            }
        } else if (isFinalParameter(componentName, parameterName) && failForFinalParameters) {
            if ( ! isValueEqualToLoadedValue(componentName, parameterName, value) ) {
                throw new IllegalArgumentException(componentName+"/"+parameterName + " is a final configuration parameter and cannot be modified at runtime");
            }
        } 
        return getParameterSet(componentName).submitChange(parameterName, value);
    }

    public Object submitChange(String componentName, String parameterName, Object value) {
        return submitChange(componentName, parameterName, value, false);
    }

    public boolean hasSubmittedChanges() {
        for (Map.Entry<String, ConfigurationHandler> e1 : componentParameterHandlers.entrySet()) {
            if (e1.getValue().hasSubmittedChanges()) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * This commit operation should operate under the control of a configuration lock.
     * @param categoriesToCommit
     * @return 
     */
    public ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit)  {
        return commitBulkChange(categoriesToCommit, false, false);
    }
    public ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit, String component)  {
        return commitBulkChange(categoriesToCommit, false, false, component);
    }
    private ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit, boolean isInit, boolean isBuild)  {
        return commitBulkChange(categoriesToCommit, isInit, isBuild, null);
    }
    
    private ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit, boolean isInit, boolean isBuild, String component)  {

        // TO-DO : should before == currentConfigurationInfo be checked ?
        ConfigurationView before = getLiveConfigurationView();
        ConfigurationView submittedChanges = new ConfigurationView();
        

        final Map<String, ConfigurationHandler> componentHandlers = new LinkedHashMap<>();
        if ( component == null ) {
            componentHandlers.putAll(componentParameterHandlers);
        } else {
            componentHandlers.put(component, componentParameterHandlers.get(component));
        }
        
        for (Map.Entry<String, ConfigurationHandler> e1 : componentParameterHandlers.entrySet()) {            
            for(Map.Entry<String, String> e : e1.getValue().getSubmittedChanges().entrySet()) {
                submittedChanges.putParameterValue(e1.getKey(), e.getKey(), e.getValue());
            }
        }
        
        log.fine("processing the following submitted changes :" + submittedChanges.getAsParameterPathMap());    
        
        // Test against the configuration validation methods
        BulkValidationException excThrown = null;
        if ( ! isBuild ) {
            try {
                for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                    ch.invokeValidateBulkChange(isInit);
                }
            } catch (Exception ex) {
                excThrown = new BulkValidationException(ex);
            } finally {
                ConfigurationView afterValidation = getLiveConfigurationView();
                ConfigurationView diff = before.diff(afterValidation);
                if (!diff.isEmpty()) {
                    dropAllSubmittedChanges();
                    Set<ParameterPath> suspect = diff.getAsParameterPathMap().keySet();
                    throw new BulkSettingException("some parameters have been modified during the validation step : " + suspect, excThrown);
                }
                if (excThrown != null) {
                    throw excThrown;
                }
            }
        }
        
        // processing the bulk change
        for (Map.Entry<String, ConfigurationHandler> entry : componentHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {                
                ch.invokeSetParameters(before.getValuesForComponent(entry.getKey()),isBuild);
            } catch (Exception ex) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("Problem setting parameter "+entry.getKey()+": "+ex.getMessage(), ex);
            }
        }
        
        ConfigurationView after = getLiveConfigurationView();
        ConfigurationView diff = before.diff(after);
        
        // Check for consistency with the original submitted changes
        for (Map.Entry<ParameterPath, String>  pathVal : diff.getAsParameterPathMap().entrySet()) {
            ParameterPath path = pathVal.getKey();
            if (isParameterReadOnly(path.getComponentName(), path.getParameterName())) {                
                continue;
            }
            if(!submittedChanges.containsPath(path)) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("the parameter " + path 
                        + " was not supposed to change from " + before.getPathValue(path)
                        + " to " + pathVal.getValue());
            } else if (!submittedChanges.getPathValue(path).equals(pathVal.getValue())) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("wrong value for parameter : " 
                        + path.toString()
                        + ", expected : "+submittedChanges.getPathValue(path)
                        + ", actual : " + pathVal.getValue());
            }
        }
        
        for (Map.Entry<String, ConfigurationHandler> entry : componentHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {
                ch.dropSubmittedChanges();
            } catch (Exception ex) {
                dropAllSubmittedChanges();
                throw new BulkSettingException(ex.getMessage(), ex);
            }
        }
        after.setConfigurationDescription(categoriesToCommit);
        return after;
    }
        
    public ConfigurationView setSingleParameter(String componentName, String parameterName, Object value) {
        return setSingleParameter(componentName, parameterName, value, false);        
    }
    public ConfigurationView setSingleParameter(String componentName, String parameterName, Object value, boolean isReadOnlyOk) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter " + new ParameterPath(componentName, parameterName));
            return new ConfigurationView();
        }
        if ( hasSubmittedChanges() ) {
            throw new RuntimeException("Cannot change single parameter with outstanding submitted changes.");
        }  

        ConfigurationView view = null;        
        try {
            submitChange(componentName, parameterName, value, isReadOnlyOk);
            view = commitBulkChange(null);        
        } catch (Exception e) {
            dropAllSubmittedChanges();
            throw e;
        }
        return view;
    }
    
    
    
    /**
     * Returns a live map of the values of the configurable parameters for the
     * given component.
     * @param componentName
     * @param categorySet
     * @return 
     */
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categorySet) {
        return getParameterSet(componentName).getCurrentValues(categorySet)
                .entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().getParameterName(), 
                        entry -> entry.getValue()));
    }

    public ConfigurationView getLiveConfigurationView()  {
        ConfigurationView res = new ConfigurationView();
        for(String compName : componentParameterHandlers.keySet()) {
            res.putValuesForComponent(compName, getCurrentValuesForComponent(compName, categories));
        }
        return res;
        
    }
    
    public Set<String> getConfigurableComponents() {
        return componentParameterHandlers.keySet();
    }
    
}
