package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import static org.lsst.ccs.bus.data.ConfigurationInfo.DEFAULT_VERSION;
import static org.lsst.ccs.bus.data.ConfigurationInfo.UNDEF_VERSION;

/**
 * A configuration summary.
 * @author emarin
 */
public class ConfigurationDescription implements Serializable {
    
    private static final long serialVersionUID = 2262886830605244842L;
    
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_INITIAL_CONFIG_NAME = "defaultInitial";
    public static final String DEFAULT_CAT = "";
    public static final String SAFE_CONFIG_NAME = "safe";
    
    /** A global name. */
    private String name = null;
    // The global version
    private int version = UNDEF_VERSION;
    
    private final Set<String> categories = new HashSet<>();
    
    private final Map<String, String> categoryTags = new HashMap<>();
    
    private final Map<String, Integer> categoryVersions = new HashMap<>();
    
    private final Map<String,CategoryTag> categoryTagsMap = new TreeMap<>();
    
    public ConfigurationDescription() {
    }
    
    public ConfigurationDescription(Set<String> categories) {
        this.categories.addAll(categories);
    }
    
    public static ConfigurationDescription safeConfiguration(Set<String> categories) {
        ConfigurationDescription res = new ConfigurationDescription(categories);
        for (String cat : categories) {
            res.putTagForCategory(cat, SAFE_CONFIG_NAME, null);
        }
        return res;
    }
    
    public ConfigurationDescription parseConfigurationString(String ... taggedCategories) {
        if (taggedCategories == null || taggedCategories.length == 0) {
            return this;
        }
        for (String s : taggedCategories){
            CategoryTag categoryTag = CategoryTag.parseCategoryTagInput(s);
//            if ( categoryTagsMap.containsKey(categoryTag.getCategoryName()) ) {
//                throw new RuntimeException("Category "+categoryTag.getCategoryName()+" already exist in this ConfigurationDescription");
//            }
            addCategoryTag(categoryTag);
        }
        return this;
    }
    
    public ConfigurationDescription withDefaults(boolean withDefaults) {
        if (withDefaults) {
            for(String cat : categories) {
                if(!categoryTagsMap.containsKey(cat)) {
                    putTagForCategory(cat, DEFAULT_CONFIG_NAME, DEFAULT_VERSION);
                }
            }
        }
        return this;
    }
    
    public ConfigurationDescription withDefaultInitial(boolean withDefaults) {
        if (withDefaults) {
            for(String cat : categories) {
                if(!categoryTagsMap.containsKey(cat)) {
                    putTagForCategory(cat, DEFAULT_INITIAL_CONFIG_NAME, DEFAULT_VERSION);
                }
            }
        }
        return this;
    }

    public void addCategoryTag(CategoryTag categoryTag) {
        String cat = categoryTag.getCategoryName();
//        if(categories != null && !categories.contains(cat)) {
//            throw new IllegalArgumentException("non existing category " + cat);
//        }
        CategoryTag newCat = new CategoryTag(categoryTag.getCategoryName()).merge(categoryTag);
        newCat.setHasChanges(categoryTag.hasChanges);
        categoryTagsMap.put(cat, newCat);
        for (String tag : categoryTag.getTags()) {
            categoryTags.put(cat, tag);
            categoryVersions.put(cat, categoryTag.getTagVersion(tag));
        }
    }
    
    
    public void putTagForCategory(String cat, String tag, Integer version) {
//        if(categories != null && !categories.contains(cat)) {
//            throw new IllegalArgumentException("non existing category " + cat);
//        }
        CategoryTag catTag = categoryTagsMap.get(cat);
        if ( catTag == null ) {
            catTag = new CategoryTag(cat);
            categoryTagsMap.put(cat, catTag);
        }
        catTag.addOrUpdateTagVersion(tag, version);
        categoryTags.put(cat, tag);
        categoryVersions.put(cat, version);
    }
    

    /**
     * Get the Set of categories in this description.
     * @return The Set with the categories.
     */
    public Set<String> getCategoriesSet() {
        return Collections.unmodifiableSet(categoryTagsMap.keySet());
    }

    public CategoryTag getCategoryTag(String category) {
//        if ( !categoryTagsMap.containsKey(category) ) {
//            throw new IllegalArgumentException("non existing category " + category+" what exists is "+categoryTagsMap.keySet());
//        }
        return categoryTagsMap.get(category);
    }

    @Deprecated
    public Map<String, String> getCategoryTags() {
        return Collections.unmodifiableMap(categoryTags);
    }
    
    @Deprecated
    public Map<String, Integer> getCategoryVersions() {
        return Collections.unmodifiableMap(categoryVersions);
    }
    
    public boolean isEmpty() {
        return categoryTagsMap.isEmpty();
    }
    
    public boolean containsCategory(String category) {
        return categoryTagsMap.containsKey(category);
    }
    
    public String getDescriptionName() {
        return "["+toString()+"]";
    }
    
    public String getName() {
        return name;
    }
    
    public int getVersion() {
        return version;
    }

    public void setName(String name, int version) {
        this.name = name;
        this.version = version;
    }

    public ConfigurationDescription merge(ConfigurationDescription input) {
        for (String category : input.getCategoriesSet() ) {
            CategoryTag inputCategoryTag = input.getCategoryTag(category);
            CategoryTag thisCategoryTag = getCategoryTag(category);
            if ( thisCategoryTag == null ) {
                categories.add(category);
                addCategoryTag(inputCategoryTag);
            } else {
                for (String tag : inputCategoryTag.getTags() ) {
//                    if ( thisCategoryTag.containsTag(tag) ) {
//                        throw new IllegalArgumentException("CategoryDescription "+thisCategoryTag+" already contains tag "+tag);
//                    }
                    thisCategoryTag.addOrUpdateTagVersion(tag, inputCategoryTag.getTagVersion(tag));
                }
            }
        }
        return this;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.categoryTagsMap);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ConfigurationDescription other = (ConfigurationDescription) obj;
        return categoryTagsMap.equals(other.categoryTagsMap);
    }
    
    public String toString() {
        if ( isEmpty() ) {
            return "";
        }
        StringBuilder sb = new StringBuilder(0); 
        for (Map.Entry<String, CategoryTag> e : categoryTagsMap.entrySet()) {
            CategoryTag catTag = e.getValue();
            sb.append(catTag.toString());
            sb.append(",");
        }
        return sb.substring(0, sb.length()-1);
    }
    
    public static ConfigurationDescription fromConfigurationInfo(ConfigurationInfo ci, Set<String> categories) {
        String desc = ci.getConfigurationDescription();
        desc = desc.replace("[", "").replace("]", "");
        ConfigurationDescription cd = new ConfigurationDescription();
        cd = cd.parseConfigurationString(desc.split(","));
        ConfigurationDescription res = new ConfigurationDescription();
        for ( String cat : cd.getCategoriesSet() ) {
            if ( categories.contains(cat) ) {
                res.addCategoryTag(cd.getCategoryTag(cat));
            }
        }
        return res;
    }

    public static ConfigurationDescription fromConfigurationInfo(ConfigurationInfo ci) {
        return fromConfigurationInfo(ci, ci.getCategorySet());
    }
}
