package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Class that describes a category tag, i.e. the particular configuration tag
 * that has been loaded for a given category.
 * 
 * The syntax for a category tag is: "categoryName:tag1|tag2|tag3"
 * where the pipe symbol "|" is used to separate the various tags that have been
 * loaded for this category.
 * 
 * Please note the following special cases:
 * 
 *  - "categoryName" means that the default tag ("") has been loaded for the given category
 *  - ":" means that the default tag ("") was loaded for the default category ("")
 *  - ":tag1" means that "tag1" was loaded for the default category ("")
 *  - "categoryName:tag1|tag2" means that "tag1" and "tag2" (in this order) have been loaded for category "categoryName"
 * 
 * @author The LSST CCS Team
 */
public class CategoryTag implements Serializable {
    
    private final String categoryName;
    private final List<String> tags = new ArrayList<>();
    private final Map<String,Integer> tagVersions = new HashMap<>();
    volatile boolean hasChanges = false;
    
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_INITIAL_CONFIG_NAME = "defaultInitial";
    public static final String DEFAULT_CAT = "";
    public static final String SAFE_CONFIG_NAME = "safe";
    private static final String DEFAULT_V_STRING = "d";
    private static final String LATEST_V_STRING = "l";
    private static final String UNDEF_V_STRING = "u";
    public static final int DEFAULT_VERSION = -1;
    public static final int LATEST_VERSION = -2;
    public static final int UNDEF_VERSION = -3;
    
    private static final Pattern CATEGORY_TAG_PATTERN = Pattern.compile("((?<category>.*?)[:])?(?<tags>.*?)(?<hasChanged>\\*?)");
    private static final Pattern TAG_PATTERN = Pattern.compile("(?<tag>.*?)([(]{1}?((?<version>[0-9]*|l|d|u))[)]{1})?");
        
    public CategoryTag(String categoryName) {
        this.categoryName = categoryName;
    }

    public String getCategoryName() {
        return categoryName;
    }
    
    public boolean containsTag(String tag) {
        return tags.contains(tag);
    }
    
    public Integer getTagVersion(String tag) {
        if ( ! containsTag(tag) ) {
            throw new IllegalArgumentException("Tag \""+tag+"\" does not exist for this CategoryTag");
        }
        return tagVersions.get(tag);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getCategoryName()).append(":");
        int nTags = getNumberOfTags();
        for(String tag : getTags()) {
            nTags--;
            sb.append(tag).append("(").append(getTagVersionAsString(tag)).append(")");
            if (nTags > 0) {
                sb.append("|");
            }
        }
        if ( hasChanges ) {
            sb.append("*");
        }
        return sb.toString();
    }

    public void setHasChanges(boolean hasChanges) {
        this.hasChanges = hasChanges;
    }
    
    private String getTagVersionAsString(String tag) {
        if ( ! containsTag(tag) ) {
            throw new IllegalArgumentException("Tag \""+tag+"\" does not exist for this CategoryTag");
        }
        Integer ver = tagVersions.get(tag);
        if (ver == null) {
            return UNDEF_V_STRING;
        }
        switch (ver) {
            case UNDEF_VERSION:
                return UNDEF_V_STRING;
            case LATEST_VERSION:
                return LATEST_V_STRING;
            case DEFAULT_VERSION:
                return DEFAULT_V_STRING;
            default:
                return String.valueOf(ver);
        }
    }
    
    
    public List<String> getTags() {
        return (tags);
    }
    
    public void addOrUpdateTagVersion(String tag, Integer version) {
        if ( !tags.contains(tag) ) {
            tags.add(tag);
        }
        tagVersions.put(tag, version);
    }

    private void addTagVersion(String tag, Integer version) {
        if ( tags.contains(tag) ) {
            throw new RuntimeException("Tag: "+tag+" already exists for this CategoryTag");
        }
        addOrUpdateTagVersion(tag, version);
    }


    
    int getNumberOfTags() {
        return tags.size();
    }
    
    public CategoryTag merge(CategoryTag categoryTag) {
        for(String tag : categoryTag.getTags()) {
            addOrUpdateTagVersion(tag, categoryTag.getTagVersion(tag));
        }
        return this;
    }
    
    public static final CategoryTag parseCategoryTagInput(String categoryTagInput) {
        Matcher m = CATEGORY_TAG_PATTERN.matcher(categoryTagInput);
        if (m.matches()) {
            String cat = m.group("category");
            if ( cat == null ) {
                cat = DEFAULT_CAT;
            }
            CategoryTag categoryTag = new CategoryTag(cat);
            String allTags = m.group("tags");
            String[] tags = allTags == null ? new String[]{DEFAULT_CONFIG_NAME} : allTags.split("\\|");
            
            boolean hasChanged = "*".equals(m.group("hasChanged"));
            categoryTag.setHasChanges(hasChanged);
                        
            for (String t : tags) {
                Matcher tagMatcher = TAG_PATTERN.matcher(t);
                if ( tagMatcher.matches() ) {

                    String tag = tagMatcher.group("tag");
                    if (tag == null) {
                        tag = DEFAULT_CONFIG_NAME;
                    }
                    String v = tagMatcher.group("version");
                    int ver;
                    if (v == null || v.isEmpty() || v.equals(DEFAULT_V_STRING)) {
                        ver = DEFAULT_VERSION;
                    } else if (v.equals(LATEST_V_STRING)) {
                        ver = LATEST_VERSION;
                    } else if (v.equals(UNDEF_V_STRING)) {
                        ver = UNDEF_VERSION;
                    } else {
                        ver = Integer.valueOf(v);
                    }
                    categoryTag.addTagVersion(tag, ver);
                } else {
                    throw new IllegalArgumentException("Could not match tag: "+t);
                }
            }
            return categoryTag;
        }
        throw new IllegalArgumentException("Could not parse a CategoryTag from input string: \""+categoryTagInput+"\"");
    }
    
    
}
