package org.lsst.ccs.commons.annotations.scanner;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * An object that scans a given object's fields for the annotation of type {@code T}.
 * TO-DO : keep a cache between successive invocations of {@code scan}
 * @author LSST CCS Team
 * @param <T> the class of the annotation to scan.
 */
public class FieldAnnotationScanner<T extends Annotation> {

    private final Class<T> annotationCls;
    
    public FieldAnnotationScanner(Class<T> annotation) {
        this.annotationCls = annotation;
    }
    
    /**
     * Scans the given object for the field annotation. Subclasses up to
     * Object.class are scanned as well.
     *
     * @param target the object to scan.
     * @return null if no annotations were found, a {@code ReflectObject} that
     * represents the object with regards to the annotation otherwise.
     */
    public ReflectObject<T> scan(Object target) {
        ReflectObjectImpl<T> soi = new ReflectObjectImpl<T>(target);
        Class cls = target.getClass();
        while (cls != Object.class) {
            Field[] fields = cls.getDeclaredFields();
            for (Field field : fields) {
                if (field.isAnnotationPresent(annotationCls)) {
                    soi.fields.put(field.getName(), field);
                    soi.annotations.put(field.getName(), field.getAnnotation(annotationCls));
                }
            }
            cls = cls.getSuperclass();
        }
        if (soi.getFields().isEmpty()) {
            return null;
        } 
        return (soi.getFields().isEmpty()) ? null : soi;
    }
    
    private class ReflectObjectImpl<T extends Annotation> implements ReflectObject<T> {

        private ReflectObjectImpl(Object target) {
            this.target = target;
        }
        
        private final Object target;
        
        private final Map<String, Field> fields = new HashMap<>();
        
        private final Map<String, T> annotations = new HashMap<>();

        @Override
        public Map<String, Field> getFields() {
            return Collections.unmodifiableMap(fields);
        }

        @Override
        public T getAnnotation(String fieldName) {
            return annotations.get(fieldName);
        }

        @Override
        public Object getTarget() {
            return target;
        }
        
    }
    
   
        
}
