package org.lsst.ccs.command;

import org.lsst.ccs.command.annotations.Command;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * This class consists of a CommandDictionary plus a map of which methods should
 * be called as a result of each command. It does not require a reference to any
 * particular object, just a class.
 *
 * Note that since this object is not a CommandSet, the utilities for combining
 * and working with CommandSets cannot be used with this class.
 *
 * @author tonyj
 */
public class CommandDictionaryBuilder  {

    private MethodBasedCommandDictionary dict;
    private final Map<DictionaryCommand, Method> methods = new HashMap<>();
    private static final boolean isSimulation = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.run.mode","normal").toLowerCase().equals("simulation");

    /**
     * Build a command dictionary for the given class.
     *
     * @param klass The class to build the dictionary from
     * @throws AmbiguousCommandException If ambiguity is detected.
     */
    public CommandDictionaryBuilder(Class klass) throws AmbiguousCommandException {
        init(klass);
    }

    /**
     * Get the command dictionary.
     *
     * @return The command dictionary.
     */
    public Dictionary getCommandDictionary() {
        return dict;
    }

    /**
     * Find the method which should be invoked as a result of a particular
     * command.
     *
     * @param command The command to be invoked
     * @return The method to be called to invoke the command, or
     * <code>null</code> if the command is not known.
     */
    Method getMethod(BasicCommand command) throws CommandArgumentMatchException {
        DictionaryCommand dc = dict.findCommand(command);
        if (dc == null) {
            return null;
        }
        return methods.get(dc);
    }

    private void init(Class targetClass) throws AmbiguousCommandException {
        dict = new MethodBasedCommandDictionary();
        for (Method targetMethod : targetClass.getMethods()) {
            DictionaryCommand dc = getDictionaryCommandFromMethod(targetMethod);
            if (dc != null) {
                // Check that this command is not ambiguous with an already defined command
                if (DictionaryUtils.containsDictionaryCommand(dict, dc)) {
                    throw new AmbiguousCommandException(String.format("Ambiguous command with name %s and %d arguments in class %s",
                            dc.getCommandName(), dc.getArguments().length, targetClass.getName()));
                }
                dict.add(dc);
                methods.put(dc, targetMethod);
            }
        }
    }

    static DictionaryCommand getDictionaryCommandFromMethod(Method targetMethod) {
        Method annotatedMethod = getAnnotationWithInheritance(targetMethod, Command.class);
        if (annotatedMethod != null) {
            Command annotation = annotatedMethod.getAnnotation(Command.class);
            if (annotation != null) {
                if ( ! isSimulation && annotation.simulation() ) {
                    return null;
                }
                return new MethodBasedDictionaryCommand(annotatedMethod, annotation);
            }
        }
        return null;
    }

    /**
     * Java does not currently support inheritance of method annotations (only
     * Class annotations). This method works around this be searching for
     * annotations for a particular method on super classes.
     *
     * @param method The method on which to search
     * @param annotationClass The annotation subclass to search for
     * @return The annotated method if found, or <code>null</code> if not found
     */
    private static <T extends Annotation> Method getAnnotationWithInheritance(Method method, Class<T> annotationClass) {
        for (;;) {
            T annotation = method.getAnnotation(annotationClass);
            if (annotation != null) {
                return method;
            }
            Class superClass = method.getDeclaringClass().getSuperclass();
            if (superClass == null) {
                break;
            }
            try {
                method = superClass.getMethod(method.getName(), method.getParameterTypes());
            } catch (NoSuchMethodException ex) {
                break;
            }
        }
        return null;
    }
}
