package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * A CommandSet that acts as a filter and deals with changing the active route.
 *
 * @author turri
 */
public class RouteSelectionCommandSet extends CompositeCommandSet implements AllowedValuesProvider {

    private final CompositeCommandSet parentCommandSet;
    private MethodBasedDictionaryArgument setArg;
    private String selectedRoute = "";
    private final CompositeCommandSet activeRouteCommandSet = new CompositeCommandSet();

    public RouteSelectionCommandSet(CompositeCommandSet parentCommandSet) {
        this.parentCommandSet = parentCommandSet;
        add(parentCommandSet);
        CommandSetBuilder builder = new CommandSetBuilder();
        final BuiltIns builtIns = new BuiltIns();
        final CommandSet builtInCommandSet = builder.buildCommandSet(builtIns);
        add(builtInCommandSet);

        for (DictionaryCommand setCommand : builtInCommandSet.getCommandDictionary()) {
            if (setCommand.getArguments().length == 2) {
                setArg = (MethodBasedDictionaryArgument) setCommand.getArguments()[1];
                setArg.setAllowedValuesProvder(this);
                break;
            }
        }
        add(activeRouteCommandSet);
    }

    private void updateActiveRoute() {
        activeRouteCommandSet.clear();
        if (!"".equals(selectedRoute)) {
            Map<String, RoutingCommandSet> routesMap = getAvailableRoutesMap(parentCommandSet);
            for (Entry<String, RoutingCommandSet> entry : routesMap.entrySet()) {
                String routeName = entry.getKey();
                if (routeName.equals(selectedRoute)) {
                    CommandSet c = entry.getValue().getRouteCommandSet();
                    activeRouteCommandSet.add(c);
                } else if (routeName.startsWith(selectedRoute + "/")) {
                    String newRoute = routeName.replace(selectedRoute + "/", "");
                    CommandSet c = new RoutingCommandSet(newRoute, entry.getValue().getRouteCommandSet());
                    activeRouteCommandSet.add(c);
                }
            }
        }
    }

    void setActiveRoute(String value) {
        if ("".equals(value)) {
            selectedRoute = "";
        } else {
            Map<String, RoutingCommandSet> routes = getAvailableRoutesMap(parentCommandSet);
            //The provided value can either be full or partial route.
            //If the provided value is not a valid route it might be partial
            String fullRoute = !routes.containsKey(value) && !"".equals(selectedRoute) ? selectedRoute + "/" + value : value;
            if (!routes.containsKey(fullRoute)) {
                throw new RuntimeException("Error: could not find target \"" + value + "\".");
            }
            selectedRoute = fullRoute;
        }
        updateActiveRoute();
    }
    
    String getActiveRoute() {
        return selectedRoute;
    }

    public enum SetGetCommands {

        TARGET
    };

    /* This must be public for the command invocation to work */
    public class BuiltIns {

        @Command(description = "Set the default command target")
        public void set(@Argument(name = "item") SetGetCommands what, @Argument(name = "value", defaultValue = "") String value) {
            switch (what) {
                case TARGET:
                    setActiveRoute(value);
            }
        }

        @Command(description = "Get various settings")
        public String get(@Argument(name = "item") SetGetCommands what) {
            switch (what) {
                case TARGET:
                    return getActiveRoute();
                default:
                    return null;
            }
        }

    }

    @Override
    public List<String> getAllowedValues() {
        updateActiveRoute();
        //Get allowed values from the parent command set.
        List<String> allowedValues = new ArrayList<>(getAvailableRoutesMap(parentCommandSet).keySet());
        //Then add the allowed values from the active routes.
        for (CommandSet cs : activeRouteCommandSet.getCommandSets()) {
            if ( cs instanceof RoutingCommandSet ) {
                allowedValues.add(((RoutingCommandSet)cs).getRoute());
            }
        }
        return allowedValues;
    }

    private Map<String, RoutingCommandSet> getAvailableRoutesMap(CommandSet cs) {
        Map<String, RoutingCommandSet> result = new LinkedHashMap<>();
        if (cs instanceof RoutingCommandSet) {
            String fullRoute = ((RoutingCommandSet) cs).getRoute();
            result.put(fullRoute, (RoutingCommandSet) cs);
        }
        if (cs instanceof CompositeCommandSet) {
            for (CommandSet s : ((CompositeCommandSet) cs).getCommandSets()) {
                result.putAll(getAvailableRoutesMap(s));
            }
        }
        return result;
    }

}
