package org.lsst.ccs.command;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.time.Duration;
import java.util.Arrays;

/**
 * An implementation of DictionaryCommand based on a single annotated method.
 *
 * @author turri
 */
class MethodBasedDictionaryCommand implements DictionaryCommand {

    private final String description;
    private final String[] aliases;
    private final DictionaryArgument[] params;
    private final Command.CommandType type;
    private final Command.CommandCategory category;
    private final String name;
    private final boolean hasVarArgs;
    private final int level;
    private final boolean autoAck;
    private final Duration timeout;
    private static final long serialVersionUID = -2134147357955297580L;

    /**
     * Create a command definition from a method and associated annotation.
     *
     * @param annotatedMethod The method providing the command implementation
     * @param annotation The annotation on the method
     */
    MethodBasedDictionaryCommand(Method annotatedMethod, Command annotation) {
        
        this.description = annotation.description();
        this.aliases = splitAliases(annotation.alias());
        this.type = annotation.type();
        this.category = annotation.category();
        this.name = annotation.name().isEmpty() ? annotatedMethod.getName() : annotation.name();
        this.hasVarArgs = annotatedMethod.isVarArgs();
        this.level = annotation.level();
        this.autoAck = annotation.autoAck();
        this.timeout = annotation.timeout() > 0 ? Duration.ofSeconds(annotation.timeout()) : null;

        Class[] types = annotatedMethod.getParameterTypes();
        Annotation[][] parAnnotations = annotatedMethod.getParameterAnnotations();

        params = new MethodBasedDictionaryArgument[types.length];
        Parameter[] methodParameters = annotatedMethod.getParameters();

        for (int i = 0; i < types.length; i++) {

            String parName = methodParameters[i].getName();
            String parDescription = "";
            String defaultValue = Argument.NULL;
            for (Annotation a : parAnnotations[i]) {
                if (a instanceof Argument) {
                    Argument paramAnnotation = (Argument) a;
                    parName = paramAnnotation.name().isEmpty() ? parName : paramAnnotation.name();
                    parDescription = paramAnnotation.description();
                    if (!paramAnnotation.defaultValue().equals(Argument.NULL)) {
                        defaultValue = paramAnnotation.defaultValue();
                    }
                    break;
                }
            }
            Class parameterType = hasVarArgs && i == types.length - 1 ? types[i].getComponentType() : types[i];
            params[i] = new MethodBasedDictionaryArgument(parName, parameterType, parDescription, defaultValue);
        }

    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String[] getAliases() {
        return aliases;
    }

    @Override
    public DictionaryArgument[] getArguments() {
        return params;
    }

    @Override
    public Command.CommandType getType() {
        return type;
    }

    @Override
    public Command.CommandCategory getCategory() {
        return category;
    }

    @Override
    public String getCommandName() {
        return name;
    }

    @Override
    public boolean isVarArgs() {
        return hasVarArgs;
    }

    @Override
    public int getLevel() {
        return level;
    }

    @Override
    public boolean isAutoAck() {
        return autoAck;
    }

    @Override
    public Duration getTimeout() {
        return timeout;
    }

    private String[] splitAliases(String alias) {
        return alias.length() > 0 ? alias.split("\\s?,\\s?") : NO_ALIASES;
    }

    @Override
    public String toString() {
        return "MethodBasedDictionaryCommand{"
                + "description='" + description + '\''
                + ", aliases=" + Arrays.toString(aliases)
                + ", params=" + Arrays.toString(params)
                + ", type=" + type
                + ", name='" + name + '\''
                + ", hasVarArgs=" + hasVarArgs
                + '}';
    }
}
