package org.lsst.ccs.command;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 * Provides help based on information from a command dictionary.
 *
 * @author tonyj
 */
public class HelpGenerator implements AllowedValuesProvider {

    private final Dictionary dict;
    private final PrintWriter out;
    private CommandSet cmdSet = null;
    private final HelpCommands builtIns = new HelpCommands();

    /**
     * Create a HelpGenerator
     *
     * @param out Where the output of the help commands should be sent
     * @param dict The dictionary used to provide help
     */
    public HelpGenerator(PrintWriter out, Dictionary dict) {
        this.out = out;
        this.dict = dict;
    }

    void help() {
        builtIns.help();
    }

    void help(String command) {
        builtIns.help(command);
    }
    /**
     * Return the command set containing the HelpCommands themselves.
     * @return 
     */
    public CommandSet getCommandSet() {
        if (cmdSet == null) {
            CommandSetBuilder builder = new CommandSetBuilder();
            cmdSet = builder.buildCommandSet(builtIns);

            for (DictionaryCommand setCommand : cmdSet.getCommandDictionary()) {
                if (setCommand.getArguments().length == 1) {
                    MethodBasedDictionaryArgument helpArg = (MethodBasedDictionaryArgument) setCommand.getArguments()[0];
                    helpArg.setAllowedValuesProvder(this);
                    break;
                }
            }
        }
        return cmdSet;
    }

    @Override
    public List<String> getAllowedValues() {
        List<String> allowedValues = new ArrayList<>();
        for (DictionaryCommand cmd : dict) {
            String cmdName = cmd.getCommandName();
            if (!allowedValues.contains(cmdName)) {
                allowedValues.add(cmdName);
            }

        }
        return allowedValues;
    }

    private class HelpCommands {

        @Command(description = "List available commands")
        public void help() {
            out.printf("%s", DictionaryUtils.basicHelpForDictionary(dict));
        }

        @Command(description = "Show help for a single command")
        public void help(@Argument(name = "command") String command) {
            boolean foundCommand = false;
            DictionaryHelpGenerator helpGenerator = dict.getHelpGenerator();
            for (DictionaryCommand def : dict) {
                if (def.getCommandName().equals(command)) {
                    foundCommand = true;
                    String help = DictionaryUtils.basicHelpForCommand(def);
                    int maxNameLeng = 0, maxTypeLeng = 0;
                    for (DictionaryArgument param : def.getArguments()) {
                        int leng = param.getName().length();
                        maxNameLeng = (leng > maxNameLeng) ? leng : maxNameLeng;
                        leng = param.getSimpleType().length();
                        maxTypeLeng = (leng > maxTypeLeng) ? leng : maxTypeLeng;
                    }
                    String spaces = String.format("%80s", "");
                    for (DictionaryArgument param : def.getArguments()) {
                        if ( param.getAllowedValues().size() == 1 ) {
                            continue;
                        }
                        String name = param.getName(), type = param.getSimpleType();
                        help += "    " + name + spaces.substring(0, maxNameLeng - name.length()) + "   "
                                + type + spaces.substring(0, maxTypeLeng - type.length()) + "   " + param.getDescription() + "\n";
                    }
                    if (helpGenerator != null && helpGenerator.hasHelp(def)) {
                        help = helpGenerator.modifyHelpForCommand(def, help, false);
                    }
                    out.printf("%s", help);
                }
            }
            if (!foundCommand) {
                throw new IllegalArgumentException("No help found for " + command);
            }
        }
    }
}
