package org.lsst.ccs.command;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * A class which can combine multiple command sets to form one combined command set.
 * Individual command sets can be dynamically added and removed. Command sets can
 * be nested, in that one CompositeCommandSet may include another.
 *
 * @author tonyj
 */
public class CompositeCommandSet implements CommandSet {

    private final Set<CommandSet> commands = new LinkedHashSet<>();
    private final CompositeCommandDictionary dict = new CompositeCommandDictionary();

    /**
     * Add a new command set
     *
     * @param set The command set to add.
     */
    public void add(CommandSet set) {
        //Check if there are ambiguous commands between the dictionary being added
        //and the existing ones
        for (DictionaryCommand newCommand : set.getCommandDictionary()) {
            if (DictionaryUtils.containsDictionaryCommand(dict, newCommand)) {
                throw new AmbiguousCommandException(String.format("Ambiguous command with name %s and %d arguments ",
                        newCommand.getCommandName(), newCommand.getArguments().length));
            }
        }
        commands.add(set);
        dict.add(set.getCommandDictionary());
    }

    /**
     * Remove a command set
     *
     * @param set The command set to be removed.
     */
    public void remove(CommandSet set) {
        commands.remove(set);
        dict.remove(set.getCommandDictionary());
    }

    void clear() {
        commands.clear();
        dict.clear();
    }

    /**
     * Returns the list of command sets.
     *
     * @return An unmodifiable collection of command sets
     */
    public Set<CommandSet> getCommandSets() {
        return Collections.unmodifiableSet(commands);
    }

    /**
     * Get the dictionary associated with this command set. If there are
     * multiple command sets included in this composite command set the
     * dictionary will include entries for all of the commands in all of the
     * included command sets.
     *
     * @return The combined dictionary
     */
    @Override
    public Dictionary getCommandDictionary() {
        return dict;
    }

    /**
     * Invoke a command included in this command set.
     *
     * @param command The Command to be invoked
     * @return The result of executing the command, or <code>null</code> if
     * the command does not return a result.
     * @throws CommandInvocationException If the command fails, either because it cannot
     * be found, the arguments do not match the expected arguments, or the command fails
     * during invocation.
     * @throws org.lsst.ccs.command.CommandArgumentMatchException if one of the provided argument values does not match the allowed values.
     */
    @Override
    public Object invoke(BasicCommand command) throws CommandInvocationException, CommandArgumentMatchException {
        ArrayList<CommandArgumentMatchException> exceptions = new ArrayList<>();
        for (CommandSet set : commands) {
            try {
                if (set.getCommandDictionary().containsCommand(command)) {
                    return set.invoke(command);
                }
            } catch (CommandArgumentMatchException ex) {
                exceptions.add(ex);
            }
        }
        CommandArgumentMatchException.throwExceptionIfNeeded(exceptions);
        throw new CommandInvocationException("Error: No handler found for command %s with %d arguments", command.getCommand(), command.getArgumentCount());
    }

}
