/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.imagehandling;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.Fits;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.util.AsciiFuncs;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.imagehandling.CommandExecutor;
import org.lsst.ccs.subsystem.imagehandling.ImageHandlingConfig;
import org.lsst.ccs.subsystem.imagehandling.config.WaitForHeaderService;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import org.lsst.ccs.utilities.image.FitsCheckSum;
import org.lsst.ccs.utilities.location.Location;

class PostImageFileHandling {
    private static final Logger LOG = Logger.getLogger(PostImageFileHandling.class.getName());
    private final ImageHandlingConfig config;
    private final AtomicBoolean isHeaderServiceEnabled;
    private final Map<ImageName, CompletableFuture<List<ImageHeaderData.Header>>> futureHeaderDataForImage = new ConcurrentHashMap<ImageName, CompletableFuture<List<ImageHeaderData.Header>>>();
    private final Map<ImageName, CompletableFuture<Double>> futureMeasuredShutterTimeForImage = new ConcurrentHashMap<ImageName, CompletableFuture<Double>>();
    private final CommandExecutor commandExecutor;
    private final AlertService alertService;

    PostImageFileHandling(ImageHandlingConfig config, CommandExecutor commandExecutor, AlertService alertService) {
        this.config = config;
        this.isHeaderServiceEnabled = new AtomicBoolean(true);
        this.commandExecutor = commandExecutor;
        this.alertService = alertService;
    }

    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName) {
        boolean needToWaitForHeaderService = this.config.getWaitForHeaderService() == WaitForHeaderService.ALWAYS || this.config.getWaitForHeaderService() == WaitForHeaderService.AUTO && this.isHeaderServiceEnabled.get();
        needToWaitForHeaderService = needToWaitForHeaderService && imageName.getController() != Controller.CCS;
        boolean needToWaitForMeasuredShutterTime = imageName.getSource() == Source.MainCamera;
        return this.handleAsynchronousData(filesToProcess, imageName, needToWaitForHeaderService, needToWaitForMeasuredShutterTime);
    }

    CompletableFuture<FileList> handleAsynchronousData(CompletableFuture<FileList> filesToProcess, ImageName imageName, boolean needToWaitForHeaderService, boolean needToWaitForMeasuredShutterTime) {
        CompletionStage<FileList> resultingFileList = filesToProcess;
        if (needToWaitForHeaderService) {
            CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = this.getHeaderDataFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureHeaderData, (fl, headerData) -> this.mergeHeaderServiceData(imageName, (FileList)fl, (List<ImageHeaderData.Header>)headerData));
        }
        if (needToWaitForMeasuredShutterTime) {
            CompletableFuture<Double> futureMeasuredShutterTimeData = this.getMeasuredShutterTimeFutureForImage(imageName);
            resultingFileList = resultingFileList.thenCombine(futureMeasuredShutterTimeData, (fl, measuredShutterTime) -> this.mergeShutterTimeData(imageName, (FileList)fl, (Double)measuredShutterTime));
        }
        return resultingFileList;
    }

    FileList mergeHeaderServiceData(ImageName imageName, FileList fl, List<ImageHeaderData.Header> headerData) {
        String message = null;
        if (headerData == null) {
            message = String.format("Header service data for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                this.fixupFitsFiles(fl, headerData, false);
            }
            catch (IOException | FitsException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && this.alertService != null && this.config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert("missingHeaders", "Missing or malformed header service data");
            this.alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;
    }

    FileList mergeShutterTimeData(ImageName imageName, FileList fl, Double measuredShutterTime) {
        String message = null;
        if (measuredShutterTime == null) {
            message = String.format("The measured shutter time for %s did not arrive in a timely manner", imageName);
            LOG.log(Level.WARNING, message);
        } else {
            try {
                ArrayList<ImageHeaderData.Header> changedHeaders = new ArrayList<ImageHeaderData.Header>();
                ImageHeaderData.Header header = new ImageHeaderData.Header("SHUTTIME", measuredShutterTime, "Shutter Exposure Time");
                changedHeaders.add(header);
                if (measuredShutterTime > 0.0) {
                    ImageHeaderData.Header header2 = new ImageHeaderData.Header("XPOSURE", measuredShutterTime, "Effective exposure duration");
                    changedHeaders.add(header2);
                }
                this.fixupFitsFiles(fl, changedHeaders, true);
            }
            catch (IOException | FitsException x) {
                message = "Error while fixing FITS headers";
                LOG.log(Level.WARNING, message, x);
            }
        }
        if (message != null && this.alertService != null && this.config.getRaiseAlertOnMissingOrBadHeaderServiceData()) {
            Alert alert = new Alert("missingHeaders", "Missing measured shutter data");
            this.alertService.raiseAlert(alert, AlertState.ALARM, message);
        }
        return fl;
    }

    void handleFitsFileCommands(CompletableFuture<FileList> filesToProcess, Location location, ImageName imageName, String mode) {
        if (!this.config.getCommands().isEmpty()) {
            filesToProcess.thenAccept(fl -> {
                HashMap<String, String> env = new HashMap<String, String>();
                env.put("BOARD", location.getBoardName());
                env.put("RAFT", location.getRaftName());
                env.put("IMAGENAME", imageName.toString());
                env.put("MODE", mode);
                env.put("SEQNO", imageName.getNumberString());
                env.put("DATE", imageName.getDateString());
                env.put("CONTROLLER", imageName.getController().getCode());
                env.put("SOURCE", imageName.getSource().getCode());
                this.commandExecutor.executeFitsFileList((FileList)fl, (Map<String, String>)env, location, imageName);
            });
        }
    }

    private void fixupFitsFiles(FileList fileList, List<ImageHeaderData.Header> headerData, boolean overwrite) throws FitsException, IOException {
        HashMap<String, List> warningsDifferMap = new HashMap<String, List>();
        HashMap<String, List> warningsNullCardMap = new HashMap<String, List>();
        for (File file : fileList) {
            Fits fits = new Fits(file);
            BasicHDU hdu = fits.getHDU(0);
            Header header = hdu.getHeader();
            long deltaCheckSum = 0L;
            for (ImageHeaderData.Header headerServiceHeader : headerData) {
                CardValues cardValues;
                String newValueAsString;
                String keyword = headerServiceHeader.getKeyword();
                if (keyword == null || "COMMENT".equals(keyword)) continue;
                if (keyword.startsWith("HIERARCH")) {
                    keyword = keyword.replace(" ", ".");
                }
                HeaderCard card = header.findCard(keyword);
                Object newValue = headerServiceHeader.getValue();
                String string = newValueAsString = newValue == null ? null : newValue.toString();
                if (card != null) {
                    CardValues cardValues2;
                    List list;
                    String oldValue = card.getValue();
                    if ((oldValue == null || oldValue.isEmpty() || overwrite) && newValue != null) {
                        long oldCheckSum = FitsCheckSum.checksum((byte[])AsciiFuncs.getBytes((String)card.toString()));
                        if (newValue instanceof String) {
                            String newValueString = (String)newValue;
                            if (!newValueString.isEmpty()) {
                                card.setValue(newValueString);
                            }
                        } else if (newValue instanceof Double) {
                            Double newValueDouble = (Double)newValue;
                            card.setValue(newValueDouble.doubleValue());
                        } else if (newValue instanceof Boolean) {
                            Boolean newValueBoolean = (Boolean)newValue;
                            card.setValue(newValueBoolean.booleanValue());
                        } else if (newValue instanceof Integer) {
                            Integer newValueInteger = (Integer)newValue;
                            card.setValue(newValueInteger.intValue());
                        } else {
                            LOG.log(Level.WARNING, "Header service keyword {0} of unexpected type {1} ignored", new Object[]{keyword, newValue.getClass()});
                        }
                        long newCheckSum = FitsCheckSum.checksum((byte[])AsciiFuncs.getBytes((String)card.toString()));
                        deltaCheckSum += newCheckSum - oldCheckSum;
                        continue;
                    }
                    Class cardType = card.valueType();
                    boolean differ = true;
                    if (oldValue != null && newValue != null) {
                        oldValue = oldValue.trim();
                        if (cardType == Boolean.class) {
                            differ = oldValue.toLowerCase().charAt(0) != newValueAsString.toLowerCase().charAt(0);
                        } else {
                            boolean bl = differ = !Objects.equals(oldValue, newValueAsString);
                        }
                    }
                    if (!differ || (list = warningsDifferMap.computeIfAbsent(keyword, k -> new ArrayList())).contains(cardValues2 = new CardValues(keyword, newValueAsString, oldValue))) continue;
                    list.add(cardValues2);
                    continue;
                }
                List list = warningsNullCardMap.computeIfAbsent(keyword, k -> new ArrayList());
                if (list.contains(cardValues = new CardValues(keyword, newValueAsString, null))) continue;
                list.add(cardValues);
            }
            FitsCheckSum.updateCheckSum((Header)header, (long)deltaCheckSum);
            header.rewrite();
        }
        if (!warningsDifferMap.isEmpty() || !warningsNullCardMap.isEmpty()) {
            StringBuilder sb = new StringBuilder("The following header cards inconsistencies have been detected between the CCS value and the header service provided value:\n");
            if (!warningsDifferMap.isEmpty()) {
                sb.append("\nThe following header keywords have different values:\n");
                for (List list : warningsDifferMap.values()) {
                    for (CardValues cv : list) {
                        sb.append("   ").append(cv.toString()).append("\n");
                    }
                }
            }
            if (!warningsNullCardMap.isEmpty()) {
                sb.append("\nThe following header keywords are null in CCS:\n");
                for (List list : warningsNullCardMap.values()) {
                    for (CardValues cv : list) {
                        sb.append("   ").append(cv.toString()).append("\n");
                    }
                }
            }
            sb.append("\n");
            LOG.log(Level.INFO, sb.toString());
        }
    }

    public static void main(String[] a) throws Exception {
        HeaderCard hc = new HeaderCard("test", Boolean.FALSE.booleanValue(), "");
        Boolean b = Boolean.TRUE;
        System.out.println(hc.getValue() + " " + hc.valueType() + " ");
    }

    private boolean isString(String string) {
        if ("true".equalsIgnoreCase(string) || "false".equalsIgnoreCase(string)) {
            return false;
        }
        try {
            Double.valueOf(string);
            return false;
        }
        catch (NumberFormatException x) {
            return true;
        }
    }

    void headerDataArrived(ImageName imageName, List<ImageHeaderData.Header> headers) {
        CompletableFuture<List<ImageHeaderData.Header>> futureHeaderData = this.getHeaderDataFutureForImage(imageName);
        futureHeaderData.complete(headers);
    }

    void measuredShutterTimeArrived(MeasuredShutterTime shutterTime) {
        CompletableFuture<Double> futureMeasuredShutterTime = this.getMeasuredShutterTimeFutureForImage(shutterTime.getImageName());
        futureMeasuredShutterTime.complete(shutterTime.getMeasuredShutterOpenTime());
    }

    private CompletableFuture<List<ImageHeaderData.Header>> getHeaderDataFutureForImage(ImageName imageName) {
        return this.futureHeaderDataForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Object>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15).intValue(), TimeUnit.SECONDS));
    }

    private CompletableFuture<Double> getMeasuredShutterTimeFutureForImage(ImageName imageName) {
        return this.futureMeasuredShutterTimeForImage.computeIfAbsent(imageName, s -> new CompletableFuture<Object>().completeOnTimeout(null, Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.headerTimeoutSeconds", 15).intValue(), TimeUnit.SECONDS));
    }

    void setHeaderServiceEnabled(boolean enabled) {
        this.isHeaderServiceEnabled.set(enabled);
    }

    CompletableFuture<File> handleAdditionalFile(AdditionalFile additionalFile) {
        if (!this.config.getAdditionalFileCommands().isEmpty()) {
            HashMap<String, String> props = new HashMap<String, String>();
            props.put("FileName", additionalFile.getFileName());
            props.put("FileType", additionalFile.getFileType());
            ImageName obsId = additionalFile.getObsId();
            props.put("ImageName", obsId.toString());
            props.put("ImageDate", obsId.getDateString());
            props.put("ImageNumber", obsId.getNumberString());
            props.put("ImageController", obsId.getController().getCode());
            props.put("ImageSource", obsId.getSource().getCode());
            props.put("SEQNO", obsId.getNumberString());
            props.put("DATE", obsId.getDateString());
            props.put("CONTROLLER", obsId.getController().getCode());
            props.put("SOURCE", obsId.getSource().getCode());
            props.put("MODE", "OTHER");
            File file = this.config.getAdditionalFileName(additionalFile, props);
            try {
                try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));){
                    additionalFile.writeFile(out);
                }
                Map<String, String> env = props.entrySet().stream().collect(Collectors.toMap(e -> ((String)e.getKey()).toUpperCase(), e -> (String)e.getValue()));
                return this.commandExecutor.executeAdditional(file, obsId, additionalFile.getFileType(), env);
            }
            catch (IOException x) {
                LOG.log(Level.SEVERE, "Error handling additional file: " + additionalFile.getFileName(), x);
                return CompletableFuture.failedFuture(x);
            }
        }
        return CompletableFuture.completedFuture(null);
    }

    private static class CardValues {
        private final String serviceValue;
        private final String ccsValue;
        private final String headerKeyword;

        CardValues(String headerKeyword, String serviceValue, String ccsValue) {
            this.serviceValue = serviceValue;
            this.ccsValue = ccsValue;
            this.headerKeyword = headerKeyword;
        }

        public boolean equals(Object obj) {
            CardValues in = (CardValues)obj;
            if (!Objects.equals(in.serviceValue, this.serviceValue)) {
                return false;
            }
            if (!Objects.equals(in.headerKeyword, this.headerKeyword)) {
                return false;
            }
            return Objects.equals(in.ccsValue, this.ccsValue);
        }

        public int hashCode() {
            int hash = 7;
            hash = 73 * hash + Objects.hashCode(this.serviceValue);
            hash = 73 * hash + Objects.hashCode(this.ccsValue);
            hash = 173 * hash + Objects.hashCode(this.headerKeyword);
            return hash;
        }

        public String toString() {
            return this.headerKeyword + "\t header_service: " + this.serviceValue + "\t ccs: " + this.ccsValue;
        }
    }
}

