package org.lsst.ccs.bus.data;

import com.sun.management.OperatingSystemMXBean;
import java.io.Serializable;
import java.lang.management.ManagementFactory;
import org.lsst.ccs.bus.annotations.DataAttributes;

/**
 * A Class to capture the Health of the running process.
 * 
 * @author The LSST CCS Team
 * 
 */
public final class RuntimeInfo implements Serializable {
    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -72154637904826248L;
    
    private static final long MEGABYTE = 1000000L; // Our MB is 10**6 bytes
    
    @DataAttributes( description = "Number of Threads allocated by the CCS process", units = "unitless")
    private final int nThreads;

    @DataAttributes( description = "Maximum Memory available to the JVM", units = "MB")
    private final long maxMemory = Runtime.getRuntime().maxMemory()/MEGABYTE;

    @DataAttributes( description = "Total Memory allocated by JVM", units = "MB")
    private final long totalMemory;

    @DataAttributes( description = "Unused Memory allocated by the JVM", units = "MB")
    private final long freeMemory;

    @DataAttributes( description = "Memory used by the JVM", units = "MB")
    private final long usedMemory;

    @DataAttributes( description = "Available Memory that could be used by the JVM", units = "MB")
    private final long availableMemory;

    @DataAttributes( description = "Recent cpu usage for the whole system. This value is a double in the [0.0,1.0] interval.", units = "unitless")
    private final double processCpuLoad;

    @DataAttributes( description = "Recent cpu usage for the JVM process. This value is a double in the [0.0,1.0] interval.", units = "unitless")
    private final double systemCpuLoad;
    private transient static final OperatingSystemMXBean osBean = ManagementFactory.getPlatformMXBean(OperatingSystemMXBean.class);
    private transient static final Runtime runtime = Runtime.getRuntime();

    @DataAttributes( description = "Number of seconds since the CCS process was started", units = "s")
    private final long uptime;

    public RuntimeInfo() {
        this(0);
    }
    
    public RuntimeInfo(long uptime) {
        this.nThreads = ManagementFactory.getThreadMXBean().getThreadCount();
        this.freeMemory = runtime.freeMemory()/MEGABYTE;
        this.totalMemory = runtime.totalMemory()/MEGABYTE;
        this.usedMemory = totalMemory - freeMemory;
        this.availableMemory = maxMemory - usedMemory;
        this.processCpuLoad = osBean.getProcessCpuLoad();
        this.systemCpuLoad = osBean.getCpuLoad();
        this.uptime = uptime;
    }
    

    /**
     * Get the number of threads currently used.
     * @return The number of threads.
     */
    public int getNThreads() {
        return nThreads;
    }

    /**
     * Get the maximum memory allocated for the JVM in Megabytes.
     * @return The maximum memory in Megabytes.
     */
    public long getMaxMemory() {
        return maxMemory;
    }

    /**
     * Get the total memory currently used by the JVM in Megabytes.
     * @return The total memory used in Megabytes.
     */
    public long getTotalMemory() {
        return totalMemory;
    }

    /**
     * Get the JVM free memory in Megabytes.
     * @return The free memory used in Megabytes.
     */
    public long getFreeMemory() {
        return freeMemory;
    }

    /**
     * Get memory used by the JVM in Megabytes.
     * @return The free memory used in Megabytes.
     */
    public long getUsedMemory() {
        return usedMemory;
    }

    /**
     * Get available memory that could be used by the JVM in Megabytes.
     * @return The available memory used in Megabytes.
     */
    public long getAvailableMemory() {
        return availableMemory;
    }

    /**
     * Get the process current CPU load.
     * @return The current CPU load.
     */
    public double getProcessCpuLoad() {
        return processCpuLoad;
    }

    /**
     * Get the system CPU load.
     * @return The system CPU load.
     */
    public double getSystemCpuLoad() {
        return systemCpuLoad;
    }
    
    /**
     * Get the Agent's uptime in milliseconds.
     * This is the amount of time since the Agent was created.
     * @return The subsystem uptime in milliseconds.
     */
    public long getUptime() {
        return uptime;
    }

}
