package org.lsst.ccs.bus.data;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import org.lsst.ccs.bus.messages.EmbeddedObjectDeserializationException;

/**
 * Base class for all Alerts.
 * This class provides an alertId field that is used to uniquely identify similar
 * Alert instances: if two Alert instances have the same alertId, they are equal.
 * 
 * Alerts are raised when abnormal situations occur, and they can be raised at
 * a different AlertState level.
 * 
 * Additionally the Alert class has the following field:
 * <ul>
 *  <li>description: the description of the Alert</li>
 * </ul>
 * 
 * Since this class is final it must be used as is by subsystems. This class
 * provides a way to add key-value pairs so that subsystem developers can add
 * subsystem specific information to specific alerts. 
 * The values added to the Alert must be basic Serializable object or de-serialization
 * problems will arise in the receiving clients.
 * The Alert data can be provided only at construction time and cannot be modified.
 * 
 * When an Alert is raised a StatusAlert message will be published on the Status 
 * bus.
 * 
 * @author The LSST CCS Team
 */
public final class Alert implements Serializable {

    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = -568951332963342L;

    private final String alertId, description; 
    
    private final Map<String,Object> innerData = new HashMap();
    
    /**
     * Base constructor for all Alerts.
     * 
     * @param alertId The alertId for this Alert instance.
     * @param description The Alert description
     */
    public Alert(String alertId, String description) {
        this(alertId, description, null);
    }

    /**
     * Base constructor for all Alerts.
     * 
     * @param alertId The alertId for this Alert instance.
     * @param description The Alert description
     * @param innerData The Alert data.
     */
    @Deprecated
    public Alert(String alertId, String description, Map<String,Object> innerData) {
        if ( innerData == null ) {
            innerData = Collections.EMPTY_MAP;
        }
        this.alertId = alertId;
        this.description = description;
        for (Entry<String,Object> e : innerData.entrySet()) {
            String key = e.getKey();
            Object obj = e.getValue();
            if ( obj instanceof Serializable ) {
                addAlertData(key, (Serializable) obj);
            } else {
                this.innerData.put(key,obj);
            }
        }
    }
    
    /**
     * Get the Alert id
     * @return The Alert id
     */
    public String getAlertId() {
        return alertId;
    }

    /**
     * Get the Alert description
     * @return The Alert description
     */
    public String getDescription() {
        return description;
    }
    
    /**
     * Get data contained in this Alert.
     * 
     * @param key The key with which this data was stored.
     * @return The data for the given key.
     */
    public Object getAlertData(String key) {
        Object o =  innerData.get(key);
        if ( o == null || ! (o instanceof byte[]) ) {
            return o;
        }
        ByteArrayInputStream bis = new ByteArrayInputStream((byte[]) o);
        try {
            ObjectInputStream ois = new ObjectInputStream(bis);
            return ois.readObject();
        } catch (ClassNotFoundException x) {
            throw new EmbeddedObjectDeserializationException("Class not found while deserializing payload " + key, x);
        } catch (IOException x) {
            throw new EmbeddedObjectDeserializationException("Could not deserialize payload " + key, x);
        }        
    }
    
    /**
     * Add a payload to this Alert.
     * 
     * @param key The key with which this payload is stored.
     * @param payload The payload to be added.
     * 
     */
    public void addAlertData(String key, Serializable payload) {
        if ( innerData.containsKey(key) ) {
            throw new IllegalArgumentException("Alert "+getAlertId()+" already has a payload for key "+key);
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try {
            ObjectOutputStream oos = new ObjectOutputStream(bos);
            oos.writeObject(payload);
            innerData.put(key,bos.toByteArray());
        } catch (IOException ioe) {
            throw new RuntimeException("Could not add data to Alert",ioe);
        }
    }

    /**
     * Get the Set of names for the Alert data.
     * @return The Set of Alert data names
     */
    public Set<String> getAlertDataNames() {
        return innerData.keySet();
    }

    @Override
    public String toString() {
        return "Alert "+getAlertId()+" ("+getDescription()+")";
    }


    @Override
    public int hashCode() {
        int hash = 7;
        hash = 13 * hash + Objects.hashCode(this.alertId);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (! (obj instanceof Alert) ) {
            return false;
        }
        final Alert other = (Alert) obj;
        return Objects.equals(this.alertId, other.alertId);
    }
    
}
