package org.lsst.ccs.bus.data;

import java.io.Serializable;
import org.lsst.ccs.bus.states.AlertState;

/**
 * This class represents an instance when an Alert was raised.
 * It contains the CCS timestamp of when the Alert was raised and the
 * AlertState with which it was raised. This object can stand for several raised
 * alert events if those alerts are raised at a high rate, in which case the CCS 
 * timestamp corresponds to the first time the alert was raised, and the cause to
 * the latest alert cause.
 * 
 * This class is immutable outside of the package it belongs to.
 * 
 * @author The LSST CCS Team
 * 
 */
public final class RaisedAlertInstance implements Serializable {
    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 312950371630569273L;
    
    private final AlertState severity;
    private final long timestamp;
    private final String description;
    private final int count;
    
    RaisedAlertInstance(AlertState severity, long timestamp, String cause) {
        this.severity = severity;
        this.timestamp = timestamp;
        this.description = cause;
        this.count = 1;
    }
    
    RaisedAlertInstance(AlertState severity, long timestamp, String cause, int count) {
        this.severity = severity;
        this.timestamp = timestamp;
        this.description = cause;
        this.count = count;
    }

    /**
     * The CCS timestamp of when the RaisedAlertInstance was raised.
     *
     * @return The CCS timestamp.
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * The AlertState at which the RaisedAlertInstance was raised.
     *
     * @return The AlertState of the Alert.
     */
    public AlertState getAlertState() {
        return severity;
    }
    
    /**
     * What caused this alert to be raised.
     * 
     * @return The description of this instance of the RaisedAlert.
     */
    public String getCause() {
        return description;
    }
        
    /**
     * The number of time this alert was raised.
     * @return 
     */
    public int getCount() {
        return count;
    }
}
