package org.lsst.ccs.bus.data;

import java.io.Serializable;
import java.util.Objects;
import java.util.Properties;

/**
 * A class to encapsulate an agent's name and type.
 * This class is used to represent an Agent across the buses.
 * The Agent's name is its unique name on the Buses.
 * Its type defines its role on the buses.
 * 
 * Two AgentInfo instances are considered equal if they have the same name and type.
 * When two AgentInfo instances are equal, they represent the same Agent on
 * the buses.
 *
 * @author emarin
 */
//REVIEW: Since the name is guaranteed to be unique, it seems not necessary
//to test the type for equality.
public final class AgentInfo implements Serializable {
    
    /**
     * Change when backward incompatible changes are made.
     */
    private static final long serialVersionUID = 716901596341446997L;

    private final String name;
    private final AgentType type;
    private final Properties agentProps;

    /**
     * The base constructor of an AgentInfo. It takes the Agent's name and its 
     * AgentType.
     * 
     * @param name The name of the Agent as defined on the Buses.
     * @param type The AgentType of the Agent. This determines its role.
     */
    public AgentInfo(String name, AgentType type) {
        this(name, type, new Properties());
    }

    /**
     * The base constructor of an AgentInfo. It takes the Agent's name and its 
     * AgentType.
     * 
     * @param name The name of the Agent as defined on the Buses.
     * @param type The AgentType of the Agent. This determines its role.
     * @param agentProps A set of properties for this agent.
     */
    public AgentInfo(String name, AgentType type, Properties agentProps) {
        this.agentProps = agentProps;
        this.name = name;
        this.type = type;
    }

    /**
     * The Agent's name on the Buses. This must be unique.
     * 
     * @return The name of the Agent on the Buses.
     */
    public String getName() {
        return name;
    }

    /**
     * The Agent's AgentType. This represents its role on the buses.
     * 
     * @return The AgentType of the Agent.
     */
    public AgentType getType() {
        return type;
    }

    /**
     * Get the value of the property for this Agent.
     * null is returned if the given property is not defined.
     * 
     * @param property The name of the property to retrieve.
     * @return         The value of the Agent property. null if the given 
     *                 property is not set.
     */
    public String getAgentProperty(String property) {
        return agentProps != null ? agentProps.getProperty(property, null) : null;
    }
    
    /**
     * Check if a given property is set for this Agent.
     * 
     * @param property The name of the property to check.
     * @return         true if the property is set for this Agent. false otherwise.
     */
    public boolean hasAgentProperty(String property) {
        return getAgentProperty(property) != null;
    }
    
    
    /**
     * Enumeration of known agent types.
     * 
     */
    public static enum AgentType {

        LISTENER("Listener"),
        CONSOLE("Console"),
        WORKER("Worker"),
        SERVICE("Service"),
        MCM("MCM"),
        OCS_BRIDGE("OCS bridge"),
        LOCK_MANAGER("Lock manager");

        private final String displayName;

        private AgentType(String displayName) {
            this.displayName = displayName;
        }

        /**
         * Get the name of the AgentType for display purposes.
         * 
         * @return The display name of the AgentType.
         */
        //REVIEW: this method seems to be for GUI display purposes.
        //Maybe it does not belong here.
        public String displayName() {
            return displayName;
        }
        
    }

    /**
     * Utility method to decide this AgentInfo belongs to a WORKER or a Service
     * agent
     *
     * @return true if the type is equal or above WORKER.
     */
    public boolean isAgentWorkerOrService() {
        return type.compareTo(AgentType.WORKER) >= 0;
    }
    
    @Override
    public String toString() {
        return name + ":" + type;
    }

    @Override
    public boolean equals(Object a) {
        if (!(a instanceof AgentInfo)) {
            return false;
        } else {
            AgentInfo agent = (AgentInfo) a;
            return getName().equals(agent.getName()) && getType().equals(agent.getType());
        }

    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 89 * hash + Objects.hashCode(this.name);
        hash = 89 * hash + Objects.hashCode(this.type);
        return hash;
    }

}
