package org.lsst.ccs.bootstrap;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility methods for the Bootstrap application.
 *
 * @author turri
 * 
 * This class will no longer be public
 */
class BootstrapUtils {

    protected static final String FILE_SEPARATOR = System.getProperty("file.separator");
    protected static final String PATH_SEPARATOR = System.getProperty("path.separator");
    protected static final String DIRECTORY_SEPARATOR = "|";
    protected static final String DISTRIBUTION_JARS_DIRECTORY = "share" + FILE_SEPARATOR + "java" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_RESOURCES_DIRECTORY = "etc" + FILE_SEPARATOR;
    private static final String DISTRIBUTION_LIB_DIRECTORY = "lib" + (System.getProperty("sun.arch.data.model").equals("32") ? "" : System.getProperty("sun.arch.data.model")) + FILE_SEPARATOR;
    private static final String DISTRIBUTION_JNI_DIRECTORY = DISTRIBUTION_LIB_DIRECTORY + "jni" + FILE_SEPARATOR;
    static final String APPLICATION_MAINCLASS_PROPERTY = "org.lsst.ccs.application.mainClass";
    static final String APPLICATION_EXTENDS_PROPERTY = "org.lsst.ccs.application.extends";
    static final String OBSOLETE_MAINJAR_PROPERTY = "org.lsst.ccs.application.mainJar";
    private volatile static List<String> listOfApplications = null;

    // Pattern for external resource definition
    private static final Pattern extRes_pattern = Pattern.compile("(?<scheme>[a-zA-Z]+://)?(?<resource>.*?)(?<extensions>\\[(?<list>.*)\\])?");
    
    // Pattern to substitute environment variable in user provided properties
    private static final String envVar_str_pattern = ".*(\\$\\[env\\.(.*)\\]).*";
    private static final Pattern envVar_pattern = Pattern.compile(envVar_str_pattern);

    /**
     * Environment variable that defines the resource path.
     */
    private final static String CCS_RESOURCE_PATH_ENV_VAR = "CCS_RESOURCE_PATH";
    private final static String CCS_DISTRIBUTION_PATH_ENV_VAR = "CCS_DISTRIBUTION_PATH";
    private volatile static String ccsDistributionRootDirectory = null;
    private volatile static ResourcesTree resourcesTree = null;

    private static final Logger LOG = Logger.getLogger(BootstrapUtils.class.getName());

    protected static void reset() {
        resourcesTree = null;
        ccsDistributionRootDirectory = null;
        listOfApplications = null;
    }

    /**
     * Get the ResourceTree for this Bootstrap environment. It contains all the
     * resources defined in the resource directory list.
     *
     * @return
     */
    synchronized static ResourcesTree getBootstrapResourcesTree() {
        if (resourcesTree == null) {
            resourcesTree = new ResourcesTree();
            List<String> resourceDirectories = getOrderedListOfResourceDirectories();
            List<String> additionalResourceDirectories = extractDirectoriesFromPath(getUserProvidedResourceDirectories(), null, true);

            for (final String resourceDirectory : resourceDirectories) {
                try {
                    String resourceURI = resourceDirectory;
                    if ( !resourceURI.contains(":") ) {
                        resourceURI = "file://"+resourceDirectory;
                    }
                    ExternalResourceDirectory exDir = new ExternalResourceDirectory(resourceURI);                                                    
                    URI uri = exDir.getURI();
                    Path dirPath = Paths.get(uri);
                    if (Files.exists(dirPath)) {
                        if (additionalResourceDirectories.contains(resourceDirectory)) {
                            resourcesTree.addUserResourceDirectory(uri, exDir.getExtensionsSet());
                        } else {
                            resourcesTree.addDistributionResourceDirectory(uri, exDir.getExtensionsSet());
                        }
                    } 
                } catch (Exception e) {
                    e.printStackTrace();
                    throw new RuntimeException(e);
                }
             }
        }
        return resourcesTree;
    }

    static String getUserProvidedResourceDirectories() {
        String resourceProperty = "org.lsst." + CCS_RESOURCE_PATH_ENV_VAR.replace("_", ".").toLowerCase();
        String userProvidedResourceDirectories = System.getProperty(resourceProperty);
        return userProvidedResourceDirectories == null ? System.getenv(CCS_RESOURCE_PATH_ENV_VAR) : userProvidedResourceDirectories;
    }

    static String getUserProvidedDistributionDirectories() {
        String distributionProperty = "org.lsst." + CCS_DISTRIBUTION_PATH_ENV_VAR.replace("_", ".").toLowerCase();
        String userProvidedDistributionDirectories = System.getProperty(distributionProperty);
        return userProvidedDistributionDirectories == null ? System.getenv(CCS_DISTRIBUTION_PATH_ENV_VAR) : userProvidedDistributionDirectories;
    }

    
    static class ExternalResourceDirectory {
        private final String scheme, resource, extensions, definition;
        private final Set<String> extList = new HashSet<>();
        private URI uri;

        public ExternalResourceDirectory(String resourceDir) {
            this(resourceDir,null,null);
        }
        
        
        ExternalResourceDirectory(String directoryStr, String root, String appendDirectory) {
            Matcher m = extRes_pattern.matcher(directoryStr);
            if (!m.matches()) {
                throw new RuntimeException("Illegal external resource " + directoryStr);
            }
            scheme = m.group("scheme");
            resource = m.group("resource");
            extensions = m.group("extensions");            
            if ( extensions != null ) {
                String tmpExt = extensions.replace("[", "").replace("]", "");
                String[] extArray = tmpExt.split(",");
                for (String e : extArray) {
                    extList.add(e.replace(".", ""));
                }
                directoryStr = directoryStr.replace(extensions, "");
            }
            if (!directoryStr.endsWith(FILE_SEPARATOR)) {
                directoryStr += FILE_SEPARATOR;
            }
            
            if (scheme == null || "file://".equals(scheme)) {
                if (appendDirectory != null && !"".equals(appendDirectory)) {
                    if (!appendDirectory.endsWith(FILE_SEPARATOR)) {
                        appendDirectory += FILE_SEPARATOR;
                    }
                    directoryStr += appendDirectory;
                }
                directoryStr = directoryStr.replace("~", System.getProperty("user.home"));

                //The root for relative paths is the root of the CCS distribution.
                if ( root != null ) {
                    Path p = Paths.get(root);
                    directoryStr = p.resolve(directoryStr).normalize().toString();
                }

                if (!directoryStr.endsWith(FILE_SEPARATOR)) {
                    directoryStr += FILE_SEPARATOR;
                }
            }
            definition = directoryStr;
            try {
                uri = new URI(definition);
            } catch (Exception e) {
                //Quick and dirty fix for windows
                try {
                    File f = new File(definition);
                    uri = f.toURI();                    
                } catch (Exception e1) {
                    e.printStackTrace();
                }
            }
        }

        String getScheme() {
            return scheme;
        }
        
        String getResource() {
            return resource;
        }
        
        String getExtensions() {
            return extensions;
        }
        
        String getDefinition() {
            return definition;
        }
        
        Set<String> getExtensionsSet() {
            return extList;
        }
        
        URI getURI() {
            return uri;
        }
        
    }
    
    
    /**
     * Get the ordered list of resource directories.
     *
     * The built in default is: - Distribution resources directory:
     * distribution/etc/ - System resources directory: /etc/ccs/ - User
     * resources directory: ~/.ccs/etc/
     *
     * By setting the environment variable CCS_RESOURCE_PATH, the developer can
     * change which resource directories are searched and in which order.
     *
     * When the CCS_RESOURCE_PATH environment variable is set, the distribution
     * resource directory will be still searched first, before any of the
     * directories provided by the user.
     *
     * This list is built once and its result is cached.
     *
     * @return The list of ordered directories used to search for resources.
     *
     */
    static List<String> getOrderedListOfResourceDirectories() {
        String userProvidedResourceDirs = getUserProvidedResourceDirectories();
        String userProvidedDistributionDirs = getUserProvidedDistributionDirectories();
        return getOrderedListOfResourceDirectories(userProvidedResourceDirs, userProvidedDistributionDirs, true);
    }

    /**
     * This method should do all the work of setting up the resource directories
     * as it is used in by the tests.
     *
     * @param userProvidedResourceDirs This resembles the environment variable
     * to set the list of resource directories.
     * @param userProvidedDistributions This resembles the environment variable
     * to set the list of additional distributions.
     * @return The list of ordered directories to search for resources.
     *
     */
    static List<String> getOrderedListOfResourceDirectories(String userProvidedResourceDirs, String userProvidedDistributions, boolean checkExistance) {

        String distributionResourceDirectoriy = getDistributionResourcesDirectory();

        List<String> additionaDistributionResourceDirectories = extractDirectoriesFromPath(userProvidedDistributions, DISTRIBUTION_RESOURCES_DIRECTORY, checkExistance);

        List<String> additionalResourceDirectories = extractDirectoriesFromPath(userProvidedResourceDirs, null, checkExistance);

        List<String> orderedListOfResourceDirectories = new ArrayList<>();

        // First add the additional resource directories
        for (String dir : additionalResourceDirectories) {
            orderedListOfResourceDirectories.add(dir);
        }

        // Then add the main distribution directory
        if ( !orderedListOfResourceDirectories.contains(distributionResourceDirectoriy) ) {
            orderedListOfResourceDirectories.add(distributionResourceDirectoriy);
        }

        // Finally add any additional distribution directory
        for (String dir : additionaDistributionResourceDirectories) {
            if ( !orderedListOfResourceDirectories.contains(dir) ) {
                orderedListOfResourceDirectories.add(dir);
            }
        }

        return orderedListOfResourceDirectories;
    }

    static List<String> extractDirectoriesFromPath(String directoryPath, String appendDirectory, boolean checkExistance) {
        return extractDirectoriesFromPath(BootstrapUtils.getCCSDistributionRootDirectory(), directoryPath, appendDirectory, checkExistance);
    }

    static List<String> extractDirectoriesFromPath(String root, String directoryPath, String appendDirectory, boolean checkExistance) {

        ArrayList<String> extractedDirectories = new ArrayList<>();

        if (directoryPath != null && !"".equals(directoryPath)) {
            StringTokenizer dirsToken = new StringTokenizer(directoryPath, DIRECTORY_SEPARATOR);
            while (dirsToken.hasMoreTokens()) {
                String resourceDir = dirsToken.nextToken().trim();

                ExternalResourceDirectory exDir = new ExternalResourceDirectory(resourceDir,root, appendDirectory);                                                
                extractedDirectories.add(exDir.getDefinition());
            }
        }
        return extractedDirectories;
    }
    
    
    /**
     * Get the Distribution resources directory.
     *
     * @return The resources directory in the distribution.
     */
    static String getDistributionResourcesDirectory() {
        return getDistributionResourcesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionResourcesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_RESOURCES_DIRECTORY;
    }

    /**
     * Get the lib directory in the distribution.
     *
     * @return the lib files directory.
     *
     */
    static String getDistributionLibDirectory() {
        return getDistributionLibDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionLibDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_LIB_DIRECTORY;
    }

    /**
     * Get the jni directory in the distribution.
     *
     * @return the jni files directory.
     *
     */
    static String getDistributionJniDirectory() {
        return getDistributionJniDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJniDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JNI_DIRECTORY;
    }

    /**
     * Get the jar files directory in the distribution.
     *
     * @return the jar files directory.
     *
     */
    static String getDistributionJarFilesDirectory() {
        return getDistributionJarFilesDirectory(getCCSDistributionRootDirectory());
    }

    static String getDistributionJarFilesDirectory(String distribution) {
        if (!distribution.endsWith(FILE_SEPARATOR)) {
            distribution += FILE_SEPARATOR;
        }
        return distribution + DISTRIBUTION_JARS_DIRECTORY;
    }

    /**
     * Get the root directory of the CCS distribution from which the Bootstrap
     * code has been launched.
     *
     * @return The root directory of the CCS distribution. It ends with a
     * file.separator.
     */
    static synchronized String getCCSDistributionRootDirectory() {
        if (ccsDistributionRootDirectory == null) {
            ccsDistributionRootDirectory = getDirectoryForClass(Bootstrap.getLoaderClass());
        }
        return ccsDistributionRootDirectory;
    }

    /**
     * This is to be used only in test environment.
     *
     * @param clazz
     * @return
     */
    static String getDirectoryForClass(Class clazz) {
        URL location = clazz.getProtectionDomain().getCodeSource().getLocation();
        String tmpCcsDistributionRootDirectory = "";
        try {
            Path path = Paths.get(location.toURI());
            String sourceCodeLocation = path.toString();
            if (sourceCodeLocation.endsWith(".jar")) {
                tmpCcsDistributionRootDirectory = sourceCodeLocation.substring(0, sourceCodeLocation.lastIndexOf(FILE_SEPARATOR) + 1);
            } else if (path.toFile().isDirectory()) {
                tmpCcsDistributionRootDirectory = sourceCodeLocation + FILE_SEPARATOR;
            } else {
                throw new RuntimeException("Could not process souce code location " + sourceCodeLocation + " It is neither a directory nor a jar file.");
            }
        } catch (URISyntaxException x) {
            throw new RuntimeException("Could not process souce code location ", x);
        }

        if (tmpCcsDistributionRootDirectory.endsWith(DISTRIBUTION_JARS_DIRECTORY)) {
            tmpCcsDistributionRootDirectory = tmpCcsDistributionRootDirectory.replace(DISTRIBUTION_JARS_DIRECTORY, "");
//            throw new RuntimeException("This executable is not being run from a CCS distribution directory. "+ccsDistributionRootDirectory);
        }
        return tmpCcsDistributionRootDirectory;
    }

    /**
     * Get the list of available applications for the current Bootstrap
     * configuration.
     *
     * @return The list of application names.
     *
     */
    synchronized static List<String> getBootstrapListOfApplications() {
        if (listOfApplications == null) {
            listOfApplications = new ArrayList<>();
            List<String> allPropertiesInResources = new ArrayList<>();
            for (ResourceDirectory dir : getBootstrapResourcesTree().getResourceDirectoryList()) {
                if (!dir.supportsExtension("app") && !dir.supportsExtension("properties") ) {
                    continue;
                }
                Path resourcePath = dir.getResouceDirectoryPath();
                if (Files.exists(resourcePath) && Files.isDirectory(resourcePath)) {
                    List<String> files = new ArrayList<>();
                    try {
                        Files.list(resourcePath).forEach((p) -> files.add(p.getFileName().toString()));
                    } catch (IOException ioe) {
                        ioe.printStackTrace();
                    }
                    for (String resource : files) {
                        if (resource.endsWith(".app")) {
                            String appName = resource.replace(".app", "");
                            allPropertiesInResources.add(appName);
                        }

                    }
                }
            }

            for (String propertyFile : allPropertiesInResources) {
                Properties props = ResourcesUtils.getApplicationDefinitionFile(getBootstrapResourcesTree(), propertyFile);
                String mainClass = props.getProperty(APPLICATION_MAINCLASS_PROPERTY, "");
                String extendsApp = props.getProperty(APPLICATION_EXTENDS_PROPERTY, "");
                String applicationName = propertyFile.replace(".app", "");
                if ((!mainClass.isEmpty() || !extendsApp.isEmpty()) && !listOfApplications.contains(applicationName)) {
                    listOfApplications.add(applicationName);
                }
            }
        }
        return listOfApplications;
    }

    /**
     * Get the application definition properties file for a given application
     * name.
     *
     * @param application The name of the application.
     * @return The Properties that define the application.
     */
    static Properties getApplicationDefinitionProperties(String application) {
        String oldAppName = Bootstrap.getBootstrapApplication();
        Bootstrap.setBootstrapApplication(application);    
        Properties p = BootstrapResourceUtils.getBootstrapProperties(application);
        Properties res = new Properties();
        for (String key : p.stringPropertyNames()) {
            String value = p.getProperty(key);
            res.put(key, SubstitutionTokenUtils.resolveSubstitutionTokens(value));
        }
        Bootstrap.setBootstrapApplication(oldAppName);    
        return res;
    }

    /**
     * Parse a user provided property to match environment variables specified
     * in the form ${ENV_VARIABLE}
     *
     */
    static String parseProperty(String inputProperty) {
        String outProperty = inputProperty;
        Matcher m = envVar_pattern.matcher(outProperty);
        if (m.matches()) {
            String envVarValue = System.getenv(m.group(2));
            if (envVarValue != null) {
                outProperty = outProperty.replace(m.group(1), envVarValue);
            } else {
                if (Bootstrap.isBootstrapEnvironment()) {
                    System.err.println("[WARNING] Environment variable " + m.group(2) + " is not defined.");
                }
            }
        }
        return outProperty;
    }

}
