package org.lsst.ccs.bootstrap;

import java.io.File;
import java.io.PrintStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Paths;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import org.apache.commons.cli.ParseException;
/**
 * Called by bootstrap.sh to set up the environment prior to invoking main program
 * @author turri
 */
class BootstrapEnvironmentUtils {

    private static final String sysPropFlag = "system.property.";
    private static final String jvmOptFlag = "system.option.";
    private static final String executeKey = "system.pre-execute";

    public static void main(String[] argv) throws MalformedURLException {
        String arg = argv[0];
        switch (arg) {
            case "LD_LIBRARY_PATH":
                System.out.println(getBootstrapLibraryPath(argv));
                break;
            case "JAVA_OPTS":
                System.out.println(getJavaOpts(argv));
                break;
            case "CLASSPATH":
                System.out.println(getClasspath(argv));
                break;
            case "ALL_IN_ONE":
                allInOne(argv, System.out);
                break;
            default:
                throw new IllegalArgumentException("Illegal Option " + arg);
        }
    }
    
    private static void allInOne(String[] argv, PrintStream out) throws MalformedURLException {
        Bootstrap b = createBootstrap(argv);
        out.print("export LD_LIBRARY_PATH=");
        out.println(escape(getLibraryPath()));
        out.print("CLASSPATH=");
        out.println(escape(getClasspath()));
        out.print("BOOTSTRAP_JAVA_OPTS=");
        out.println(escape(getJavaOpts("system")));
        
        //If no application is defined, there is no need to look for a script
        //to execute.
        String applicationName = Bootstrap.getBootstrapApplication();
        if ( applicationName != null ) {        
            Properties p = Bootstrap.getBootstrapApplicationProperties();
            if (p.containsKey(executeKey)) {
                String value = p.getProperty(executeKey);
                if (!value.startsWith(File.separator)) {
                    // Look for the script on the bootstrap CLASSPATH
                    URL url = BootstrapResourceUtils.getResourceURL(value);
                    if (url == null) {
                        throw new IllegalArgumentException("Pre-execute script not found: " + value);
                    }
                    value = url.getFile();
                }
                out.println("source " + value);
            }
        }
    }
    
    private static String escape(String output) {
        return "'"+output.replace("'", "''")+"'";
    }
    
    /**
     * Build the java options string to be passed to the JVM
     * @param argv the command line arguments
     * @param fileName the name of the properties to load. 
     *                 By default system.properties
     * @return 
     */
    static String getJavaOpts(String[] argv) {
        return getJavaOpts(argv,"system");
    }
    static String getJavaOpts(String[] argv, String fileName) {

        createBootstrap(argv);
        return getJavaOpts(fileName);
    }

    private static String getJavaOpts(String fileName) throws RuntimeException {
        StringBuilder output = new StringBuilder();
        Properties p = BootstrapResourceUtils.getBootstrapProperties(fileName);
        String applicationName = Bootstrap.getBootstrapApplication();
        p.put(sysPropFlag+"org.lsst.ccs.application.name", applicationName != null ? applicationName : "");

        Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(p);
        for (Object key : keys) {
            String keyStr = (String) key;
            if (keyStr.startsWith(sysPropFlag)) {
                String sysPropertyForJava = keyStr.replace(sysPropFlag, "");
                String sysPropertyValue = p.getProperty(keyStr);
                if (sysPropertyValue != null && !sysPropertyValue.isEmpty()) {
                    output.append("-D");
                    output.append(sysPropertyForJava);
                    output.append("=");
                    output.append(BootstrapUtils.parseProperty(sysPropertyValue));
                    output.append(" ");
                }
            } else if (keyStr.startsWith(jvmOptFlag)) {
                String sysOptionForJava = keyStr.replace(jvmOptFlag, "");
                output.append("-");
                output.append(sysOptionForJava);
                
                String sysOptionValue = p.getProperty(keyStr);
                if ( sysOptionValue != null && ! sysOptionValue.isEmpty() ) {
                    output.append(BootstrapUtils.parseProperty(sysOptionValue));                    
                }
                output.append(" ");
            }
        }
        return output.toString();
    }

    private static Bootstrap createBootstrap(String[] argv) throws RuntimeException {
        Bootstrap b = new Bootstrap(true);
        try {
            b.parseCommandLineArguments(argv);
        } catch (ParseException pe) {
            throw new RuntimeException(pe);
        }
        return b;
    }

    static String getClasspath(String[] argv) {

        createBootstrap(argv);
        return getClasspath();
    }

    private static String getClasspath() throws RuntimeException {
        StringBuilder output = new StringBuilder();
        URLClassLoader loader = Bootstrap.getBootstrapApplicationClassLoader();
        
        URL[] urls = loader.getURLs();
        try {
            for (URL url : urls) {
                output.append(Paths.get(url.toURI())).append(BootstrapUtils.PATH_SEPARATOR);
            }
        } catch (URISyntaxException x) {
            throw new RuntimeException(x);
        }
        
        String ret = output.toString();
        return ret.substring(0,ret.length()-1);
    }
    
    static String getBootstrapLibraryPath(String[] argv) {

        createBootstrap(argv);
        return getLibraryPath();
    }    

    private static String getLibraryPath() {
        String libraryPath = BootstrapUtils.getDistributionJniDirectory() + File.pathSeparator + BootstrapUtils.getDistributionLibDirectory();
        String existingLibraryPath = System.getenv("LD_LIBRARY_PATH");
        if (existingLibraryPath != null && !existingLibraryPath.isEmpty()) {
            libraryPath += File.pathSeparator + existingLibraryPath;
        }

        Properties p = BootstrapResourceUtils.getBootstrapProperties("system");
        String additionalLibraryPathProperty = p.getProperty("org.lsst.ccs.ld.library.path", "");            
        if (!additionalLibraryPathProperty.isEmpty()) {
            List<String> dirs = BootstrapUtils.extractDirectoriesFromPath(BootstrapUtils.getDistributionLibDirectory(), additionalLibraryPathProperty, null, false);            
            for (String dir : dirs) {
                libraryPath += File.pathSeparator + dir;
            }
        }
        return libraryPath;
    }    
}
