package org.lsst.sal.codegen;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Takes an XML document, and builds a list of ClassInfo objects
 *
 * @author tonyj, farrukh
 */
class XMLReader {

    private final CodegenMojo mojo;

    /**
     * Create an XML reader
     * @param mojo The mojo containing all of the parameters controlling the process.
     */
    XMLReader(CodegenMojo mojo) {
        this.mojo = mojo;
    }

    /**
     * Parse the given XML file, and return list of classes
     * @param builder The builder to use for parsing
     * @param xmlURL The XML url to read
     * @return The list of classes
     * @throws SAXException
     * @throws IOException 
     */
    List<ClassInfo> read(DocumentBuilder builder, URL xmlURL, String packageName) throws SAXException, IOException {
        try (InputStream in = xmlURL.openStream()) {
            // read the array - generate class names
            Document document = builder.parse(new InputSource(in));
            return read(document, packageName);
        }
    }

    private List<ClassInfo> read(Document document, String packageName) {
        List<ClassInfo> classInfoCollection = new ArrayList<>();
        Element root = document.getDocumentElement();
        if ("SALObjects".equals(root.getTagName())) {
            NodeList childNodes = root.getChildNodes();
            for (int i = 0; i < childNodes.getLength(); i++) {
                Node item = childNodes.item(i);
                if (item instanceof Element) {
                    Element element = (Element) item;
                    classInfoCollection.addAll(handleObjectsForElement(element, packageName));
                }
            }
        } else {
            // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
            classInfoCollection.addAll(handleObjectsForElement(root, packageName));
        }
        return classInfoCollection;
    }
    // read the array - generate class names
    // now start reading
    // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
    // caution this has two subtypes - Event and Telemetry
    // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
    // caution this has two subtypes - Event and State

    private List<ClassInfo> handleObjectsForElement(Element element, String packageName) throws NumberFormatException, DOMException {
        // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
        JavaFromXML.ClassType classType;
        if (element.getTagName().contains("Event")) {
            classType = JavaFromXML.ClassType.EVENT; // caution this has two subtypes - Event and State
        } else if (element.getTagName().contains("Telemetry")) {
            classType = JavaFromXML.ClassType.TELEMETRY;
        } else {
            classType = JavaFromXML.ClassType.COMMAND;
        }
        return handleObjectsOfClass(element, classType, packageName);
    }

    private List<ClassInfo> handleObjectsOfClass(Element document, JavaFromXML.ClassType classType, String packageName) throws NumberFormatException, DOMException {
        List<ClassInfo> classInfoCollection = new ArrayList<>();

        // get the actual class blocks...
        NodeList salClassNodes = document.getElementsByTagName(classType.getTag());
        // now looping over each class -
        // loop over all classes -
        for (int i = 0; i < salClassNodes.getLength(); i++) {
            // for each class you have one of these
            Map<String, ClassInfo.Variable> variableNamesAndTypesMap = new LinkedHashMap<>();
            Map<String, ClassInfo.EnumValues> enumMap = new LinkedHashMap<>();
            // Need another list to keep track of what is actually stored in the ENUM for SAL particularly camera.sal class
            //                Map<String, String> enumTypeMapSAL = new LinkedHashMap<>();
            //                Map<String, String> enumStateTypeMapSAL = new LinkedHashMap<>();
            // now find all EFDB Topics as these define the class name
            // salNodes will be read in - even blank spaces will be read in so we
            // only want to pick those that are bonafide elements
            if (salClassNodes.item(i) instanceof Element) {
                Element eClass = (Element) salClassNodes.item(i);
                // find a child -
                NodeList topicList = eClass.getElementsByTagName("EFDB_Topic");
                // should only be ONE topic (ie className) per class
                if (topicList.getLength() > 1) {
                    // TODO: throw exception
                    System.out.println("More than one EFDB_Topic(Class Name) in class - incorrect xml \n");
                    System.exit(0);
                }
                String className = topicList.item(0).getTextContent().trim();
                String rawClassName = className.split("_", 2)[0] + "." + className.split("_", 2)[1];
                // split it up grab the last bit which is the class name
                className = className.split("_", 2)[1];
                className = className.replace("logevent_", "");
                className = className.replace("command_", "");
                if (className.contains("ccs")) {
                    className = className.replace("ccs", "CCS");
                } else {
                    // otherwise just capitalize className's first character -
                    className = className.substring(0, 1).toUpperCase() + className.substring(1);
                }
                // State Events are treated differently
                JavaFromXML.ClassType thisClassType = classType;
                if (className.endsWith("State")) {
                    thisClassType = JavaFromXML.ClassType.STATE;
                }
                // find the list of items ...
                NodeList itemList = eClass.getElementsByTagName("item");
                for (int iI = 0; iI < itemList.getLength(); iI++) {
                    if (itemList.item(iI) instanceof Element) {
                        // fine obly real items
                        Element eI = (Element) itemList.item(iI);
                        // there should only ever be ONE per item of the following
                        NodeList efdbNameList = eI.getElementsByTagName("EFDB_Name");
                        NodeList idlTypeList = eI.getElementsByTagName("IDL_Type");
                        NodeList idlSizeList = eI.getElementsByTagName("IDL_Size");
                        NodeList enumerationList = eI.getElementsByTagName("Enumeration");
                        NodeList countList = eI.getElementsByTagName("Count");
                        NodeList isJavaArrayList = eI.getElementsByTagName("IsJavaArray");
                        // bail out if someone has given us an incorrect xml ....
                        if ((efdbNameList.getLength() > 1 || idlTypeList.getLength() > 1 || enumerationList.getLength() > 1) || (efdbNameList.getLength() != idlTypeList.getLength())) {
                            System.out.println(" There should only be 1 EFDB_Name, IDL_Type, Enumeration List in an item \n ");
                            System.out.println(" Every EFDB_Name should have a corresponding IDL_Type");
                            System.exit(0);
                        }
                        int count = Integer.parseInt(countList.item(0).getTextContent().trim());
                        boolean isJavaArray = count > 1 || isJavaArrayList.getLength() > 0;
                        int size = 1;
                        if (idlSizeList.getLength() >= 1) {
                            size = Integer.parseInt(idlSizeList.item(0).getTextContent().trim());
                        }
                        // summary state fix
                        if (className.endsWith("SummaryState")) {
                            enumMap.put(efdbNameList.item(0).getTextContent().trim(), new ClassInfo.EnumValues("SummaryState", "DISABLED,ENABLED,FAULT,OFFLINE,STANDBY"));
                        }
                        if (enumerationList.getLength() > 0) {
                            String enumType = efdbNameList.item(0).getTextContent().trim();
                            enumMap.put(efdbNameList.item(0).getTextContent().trim(), new ClassInfo.EnumValues(enumType, enumerationList.item(0).getTextContent().trim()));
                            variableNamesAndTypesMap.put(efdbNameList.item(0).getTextContent().trim(), new ClassInfo.Variable(enumType, count, size, isJavaArray));
                        } else if (efdbNameList.getLength() > 0 && enumerationList.getLength() < 1) {
                            String type = idlTypeList.item(0).getTextContent().trim();
                            //silly fix for value boolean - need to exclude it from command files until xml is fixed - Farrukh
                            if (classType.equals(JavaFromXML.ClassType.COMMAND) && efdbNameList.item(0).getTextContent().trim().equals("value") && type.equals("boolean")) {
                                //do nothing
                                //System.out.println("boolean variable value found for Command Class - thrown out");
                            } else {
                                variableNamesAndTypesMap.put(efdbNameList.item(0).getTextContent().trim(), new ClassInfo.Variable(type, count, size, isJavaArray));
                            }
                        }
                    }
                }
                classInfoCollection.add(new ClassInfo(rawClassName, className, mojo, packageName, variableNamesAndTypesMap, enumMap, thisClassType, rawClassName));
            }
        } // end of loop over SAL classes
        return classInfoCollection;
    }
}
