package org.lsst.sal.codegen;

import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.TypeVariableName;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import javax.lang.model.element.Modifier;
import org.lsst.sal.SALEnum;

/**
 * Encapsulates the information needed to generate a class. This is used by
 * JavaClassWriter, TestWriter and SalWriter
 *
 * @author tonyj
 */
final class ClassInfo {

    private final Map<String, Variable> originalVariableNamesAndTypesMap;

    static class EnumValues {

        private final String type;
        private final String values;

        public EnumValues(String type, String values) {
            this.type = type;
            this.values = values;
        }

        public String getType() {
            return type;
        }

        public String getEnumConstants() {
            return values;
        }
    }

    static class Variable {

        private final int count;
        private final int size;
        private final String type;
        private final boolean isJavaArray;

        public Variable(String type, int count, int size, boolean isJavaArray) {
            this.type = type;
            this.count = count;
            this.size = size;
            this.isJavaArray = isJavaArray;
        }

        public int getCount() {
            return count;
        }

        public String getType() {
            return type;
        }

        public int getSize() {
            return size;
        }

        public boolean isJavaArray() {
            return isJavaArray;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final Variable other = (Variable) obj;
            if (this.isJavaArray != other.isJavaArray) {
                return false;
            }
            return Objects.equals(this.type, other.type);
        }

        @Override
        public int hashCode() {
            int hash = 7;
            hash = 23 * hash + Objects.hashCode(this.type);
            hash = 23 * hash + (this.isJavaArray ? 1 : 0);
            return hash;
        }

    }

    private final String className;
    private final Map<String, TypeName> convertedVariableNamesAndTypesMap;
    private final Map<String, TypeName> superClassArguments = new LinkedHashMap<>();
    private final Map<String, Variable> variableNamesAndCounts = new LinkedHashMap<>();
    private String packageName;
    private final String superClassName;
    private final Map<String, TypeSpec> enumerations;
    private final JavaFromXML.ClassType classType;
    private final String salClassName;
    private boolean isInherited = false;

    /**
     * Constructs a ClassInfo from information extracted from XML
     *
     * @param rawClassName
     * @param className
     * @param mojo
     * @param variableNamesAndTypesMap
     * @param enumMap
     * @param classType
     * @param salClassName
     */
    ClassInfo(String rawClassName, String className, CodegenMojo mojo, String packageName, Map<String, Variable> variableNamesAndTypesMap,
            Map<String, EnumValues> enumMap, JavaFromXML.ClassType classType, String salClassName) {
        this.className = computeClassName(className, classType);
        this.packageName = computePackageName(classType, packageName);
        this.superClassName = computeSuperClass(classType, mojo);
        this.enumerations = computeEnumerations(enumMap);
        this.originalVariableNamesAndTypesMap = variableNamesAndTypesMap;
        this.convertedVariableNamesAndTypesMap = convertToTypeMap(variableNamesAndTypesMap);
        this.classType = classType;
        this.salClassName = salClassName;
        boolean isState = classType == JavaFromXML.ClassType.STATE;
        if (isState) {
            // TODO: This assumes one and only one enum, always to be called substate
            String type = className.replace("Event", "").replace("Detailed", "");
            String values = enumMap.values().iterator().next().getEnumConstants();
            this.enumerations.clear();
            final TypeSpec.Builder enumBuilder = TypeSpec.enumBuilder(type)
                    .addModifiers(Modifier.PUBLIC)
                    .addSuperinterface(SALEnum.class);
            for (String value : values.split(",")) {
                enumBuilder.addEnumConstant(value.trim());
            }
            this.enumerations.put("substate", enumBuilder.build());
            this.convertedVariableNamesAndTypesMap.clear();
            this.convertedVariableNamesAndTypesMap.put("substate", ClassName.bestGuess(type));
        }
    }

    boolean checkIfCanInherit(List<ClassInfo> possibleSuperClasses) {
        // Look for a super class with the same name
        for (ClassInfo info : possibleSuperClasses) {
            if (info.className.equals(this.className)) {
                /// Check if variable names and types match
                if (info.originalVariableNamesAndTypesMap.equals(this.originalVariableNamesAndTypesMap)) {
                    this.packageName = info.packageName;
                    this.isInherited = true;
                    originalVariableNamesAndTypesMap.forEach((name, type) -> {
                        if (enumerations.containsKey(name)) {
                            TypeName typeName = ClassName.get(packageName + "." + this.className, JavaFromXML.capitalize(type.getType()));
                            if (type.isJavaArray()) {
                                typeName = ArrayTypeName.of(typeName);
                            }
                            this.convertedVariableNamesAndTypesMap.put(name, typeName);
                        }
                    });
                    return true;
                }
            }
        }
        return false;
    }

    public String getClassName() {
        return className;
    }

    public Map<String, TypeName> getVariableNamesAndTypesMap() {
        return convertedVariableNamesAndTypesMap;
    }

    public String getPackageName() {
        return packageName;
    }

    public String getSuperClassName() {
        return superClassName;
    }

    public Set<TypeSpec> getEnumerations() {
        return new HashSet(enumerations.values());
    }

    JavaFromXML.ClassType getClassType() {
        return classType;
    }

    private static String computeClassName(String className, JavaFromXML.ClassType classType) {
        switch (classType) {
            case STATE:
            case EVENT:
                return className + "Event";
            case TELEMETRY:
                return className + "Telemetry";
            case COMMAND:
                return className + "Command";
            default:
                return className;
        }
    }

    private static String computePackageName(JavaFromXML.ClassType classType, String basePackageName) {
        switch (classType) {
            case STATE:
                return basePackageName + ".states";
            case EVENT:
                return basePackageName + ".event";
            case TELEMETRY:
                return basePackageName + ".telemetry";
            case COMMAND:
                return basePackageName + ".command";
            default:
                return basePackageName;
        }
    }

    private static String computeSuperClass(JavaFromXML.ClassType classType, CodegenMojo mojo) {
        switch (classType) {
            case STATE:
                return mojo.getStateParent();
            case EVENT:
                return mojo.getEventParent();
            case TELEMETRY:
                return mojo.getTelemetryParent();
            case COMMAND:
            default:
                return mojo.getCommandParent();
        }
    }

    private Map<String, TypeName> convertToTypeMap(Map<String, Variable> variableNamesAndTypesMap) {
        Map<String, TypeName> result = new LinkedHashMap<>();
        variableNamesAndTypesMap.forEach((name, type) -> {
            result.put(name, mapVariableTypeToJava(type, name));
            variableNamesAndCounts.put(name, type);
        });
        return result;
    }

    private TypeName mapVariableTypeToJava(Variable variable, String name) {

        TypeName typeName;
        String type = variable.getType();
        if (type.equals("string")) {
            typeName = TypeName.get(String.class);
        } else if (type.equals("long long")) {
            typeName = TypeName.LONG;
        } else if (type.contains("long")) {
            typeName = TypeName.INT;
        } else if (enumerations.containsKey(name)) {
            typeName = ClassName.get(packageName + "." + className, JavaFromXML.capitalize(type));
        } else {
            typeName = TypeVariableName.get(type);
        }
        if (variable.isJavaArray()) {
            typeName = ArrayTypeName.of(typeName);
        }
        return typeName;
    }

    String getClassTypeName() {
        String whatClass = "";
        if (classType == JavaFromXML.ClassType.TELEMETRY) {
            whatClass = "telemetry";
        }
        if (classType == JavaFromXML.ClassType.COMMAND) {
            whatClass = "command";
        }
        if (classType == JavaFromXML.ClassType.EVENT) {
            whatClass = "event";
        }
        if (classType == JavaFromXML.ClassType.STATE) {
            whatClass = "state";
        }
        return whatClass;
    }

    String getSALClassName() {
        return salClassName;
    }

    Map<String, TypeName> getSuperClassArguments() {
        return superClassArguments;
    }

    Map<String, TypeName> getConstructorArguments() {
        Map<String, TypeName> map = new LinkedHashMap<>();
        map.putAll(superClassArguments);
        map.putAll(convertedVariableNamesAndTypesMap);
        return map;
    }

    private static Map<String, TypeSpec> computeEnumerations(Map<String, EnumValues> enumMap) {
        Map<String, TypeSpec> result = new LinkedHashMap<>();
        enumMap.forEach((name, values) -> {
            final TypeSpec.Builder enumerationBuilder = TypeSpec.enumBuilder(JavaFromXML.capitalize(values.getType()))
                    .addModifiers(Modifier.PUBLIC)
                    .addSuperinterface(SALEnum.class);
            for (String value : values.getEnumConstants().split(",")) {
                enumerationBuilder.addEnumConstant(value.trim());
            }
            result.put(name, enumerationBuilder.build());
        });
        return result;
    }

    boolean isEnumeration(String name) {
        return enumerations.containsKey(name);
    }

    boolean isArray(String name) {
        return convertedVariableNamesAndTypesMap.get(name) instanceof ArrayTypeName;
    }

    TypeSpec getEnumerationType(String name) {
        return enumerations.get(name);
    }

    int getCount(String name) {
        return variableNamesAndCounts.get(name).getCount();
    }

    int getSize(String name) {
        return variableNamesAndCounts.get(name).getSize();
    }

    String getGetter(String name) {
        TypeName type = convertedVariableNamesAndTypesMap.get(name);
        if (type == null) {
            type = TypeName.INT;
        }
        final String stem = type.toString().equalsIgnoreCase("boolean") ? "is" : "get";
        return stem + JavaFromXML.capitalize(name);
    }

    boolean isInherited() {
        return isInherited;
    }
}
