package org.lsst.sal.codegen;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.xml.sax.SAXException;

/**
 * This is a Maven plugin for building SimpleSAL classes as part of a maven
 * build. It takes as input
 * <ul>
 * <li>A set of SAL XML files, normally read from the classpath and contained in
 * the dependent SAL generated .jar files.</li>
 * <li>A set of parameters specified as configuration in the .pom file used to
 * invoke the plugin
 * </ul>
 *
 * @author tonyj
 */
@Mojo(name = "codegen", defaultPhase = LifecyclePhase.GENERATE_SOURCES)
public class CodegenMojo extends AbstractMojo {

    /**
     * Only used for unit tests. Creates a test instance of CodegenMojo
     *
     * @return
     */
    static CodegenMojo createTest() {
        // Hardwired for comcam for now
        CodegenMojo mojo = new CodegenMojo();
        mojo.basePackageName = "org.lsst.sal.cccamera";
        mojo.commandParent = "org.lsst.sal.camera.CameraCommand";
        mojo.eventParent = "org.lsst.sal.camera.CameraEvent";
        mojo.stateParent = "org.lsst.sal.camera.CameraStateChangeEvent";
        mojo.telemetryParent = "org.lsst.sal.camera.CameraTelemetry";
        mojo.salFile = "cccamera.sal";
        mojo.managerClassName = "org.lsst.sal.SAL_CCCamera";
        mojo.salMain = "CCCamera";
        mojo.xmlFiles = new String[]{"xml/CCCamera_Commands.xml", "xml/CCCamera_Events.xml", "xml/CCCamera_Telemetry.xml", "xml/CCCamera_Generics.xml"};
        mojo.inheritPackageName = "org.lsst.sal.camera";
        mojo.xmlInheritFiles = new String[]{"xml/MTCamera_Commands.xml", "xml/MTCamera_Events.xml", "xml/MTCamera_Telemetry.xml", "xml/MTCamera_Generics.xml"};
        mojo.xmlPath = null;
        return mojo;
    }

    @Parameter(property = "project", defaultValue = "${project}")
    private MavenProject project;

    @Parameter(property = "salMain", defaultValue = "SALMain")
    private String salMain;

    @Parameter(property = "outputDir", defaultValue = "${project.build.directory}/generated-sources/sal-code")
    private File outputDir;

    @Parameter(property = "outputTestDir", defaultValue = "${project.build.directory}/generated-test-sources/sal-test")
    private File outputTestDir;

    @Parameter(property = "outputResourcesDir", defaultValue = "${project.build.directory}/generated-sources/sal-resources")
    private File outputResourcesDir;

    // specify an input directory 
    @Parameter(property = "xmlPath", defaultValue = "${env.SIMPLESAL_XMLPATH}")
    private File xmlPath;

    @Parameter(property = "xmlFiles")
    private String[] xmlFiles;

    @Parameter(property = "xmlInheritFiles")
    private String[] xmlInheritFiles;

    // specify a package name - this can be defined in the pom.xml when used for code generation elsewhere...
    @Parameter(property = "basePackageName", defaultValue = "org.lsst.sal.camera")
    private String basePackageName;

    @Parameter(property = "inheritPackageName", defaultValue = "org.lsst.sal.camera")
    private String inheritPackageName;

    // specify managerClass Name 
    @Parameter(property = "managerClassName", defaultValue = "org.lsst.sal.SAL_MTCamera")
    private String managerClassName;

    // specify the parent class from which all telemetry classes derive/extend from
    @Parameter(property = "telemetryParent", defaultValue = "org.lsst.sal.SALTelemetry")
    private String telemetryParent;

    // specify the parent class from which all event classes derive/extend from
    @Parameter(property = "stateParent", defaultValue = "org.lsst.sal.SALStateChangeEvent")
    private String stateParent;

    // specify the parent class from which all command classes derive/extend from 
    @Parameter(property = "eventParent", defaultValue = "org.lsst.sal.SALEvent")
    private String eventParent;

    //specify the parent class from which all state classes derive/extend from  
    @Parameter(property = "commandParent", defaultValue = "org.lsst.sal.SALCommand")
    private String commandParent;

    //specify the name of the generated .sal file
    @Parameter(property = "salFile", defaultValue = "salFile.sal")
    private String salFile;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        this.project.addCompileSourceRoot(this.outputDir.getAbsolutePath());
        this.project.addTestCompileSourceRoot(this.outputTestDir.getAbsolutePath());
        Resource generatedResources = new Resource();
        generatedResources.setDirectory(outputResourcesDir.getAbsolutePath());
        this.project.addResource(generatedResources);

        this.outputDir.mkdirs();
        JavaFromXML javaXML = new JavaFromXML(this);
        try {
            javaXML.execute();
        } catch (IOException | ParserConfigurationException | SAXException e) {
            throw new MojoExecutionException("Could not generate Java source code!", e);
        }

        try {
            JavaMainWriter javaMain = new JavaMainWriter(this);
        } catch (IOException | ParserConfigurationException | SAXException e) {
            throw new MojoExecutionException("Could not generate Java source code!", e);
        }
    }

    List<URL> getXMLs() {
        return getXMLs(xmlFiles);
    }

    List<URL> getInheritXMLs() {
        return getXMLs(xmlInheritFiles);
    }

    private List<URL> getXMLs(String[] xmlFiles) {
        List<URL> xmlURLs = new ArrayList<>();
        if (xmlFiles != null) {
            for (String xmlFile : xmlFiles) {
                URL resource = null;
                if (xmlPath != null) {
                    File file = new File(xmlPath, xmlFile);
                    getLog().info("Using external XML file "+file+" "+file.exists());
                    if (file.exists()) {
                        try {
                            resource = file.toURI().toURL();
                        } catch (MalformedURLException ex) {
                            // No reason this should ever happen
                        }
                    }
                }
                if (resource == null) {
                    resource = this.getClass().getClassLoader().getResource(xmlFile);
                    if (resource == null) {
                        throw new RuntimeException("XML file not found: " + xmlFile);
                    }
                }
                xmlURLs.add(resource);
            }
        }
        return xmlURLs;
    }

    File getOutputDir() {
        return outputDir;
    }

    File getTestOutputDir() {
        return outputTestDir;
    }

    File getResourceOutputDir() {
        return outputResourcesDir;
    }

    String getManagerClassName() {
        return managerClassName;
    }

    String getBasePackageName() {
        return basePackageName;
    }

    public String getInheritPackageName() {
        return inheritPackageName;
    }

    String getSALFileName() {
        return salFile;
    }

    String getStateParent() {
        return stateParent;
    }

    String getEventParent() {
        return eventParent;
    }

    String getTelemetryParent() {
        return telemetryParent;
    }

    String getCommandParent() {
        return commandParent;
    }

    public String getSalMain() {
        return salMain;
    }
}
