package org.lsst.sal.codegen;

import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.TypeVariableName;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.lang.model.element.Modifier;
import org.lsst.sal.SALEnum;
import org.lsst.sal.codegen.JavaFromXML.Variable;

/**
 * Encapsulates the information needed to generate a class. This is used by
 * JavaClassWriter, TestWriter and SalWriter
 *
 * @author tonyj
 */
final class ClassInfo {

    private final String className;
    private Map<String, TypeName> variableNamesAndTypesMap;
    private final Map<String, TypeName> superClassArguments = new LinkedHashMap<>();
    private final Map<String, Variable> variableNamesAndCounts = new LinkedHashMap<>();
    private String packageName;
    private final String superClassName;
    private final Map<String, TypeSpec> enumerations;
    private final JavaFromXML.ClassType classType;
    private final String salClassName;
    private boolean isInherited = false;

    ClassInfo(String rawClassName, String className, String basePackageName, String stateParent, String eventParent,
            String commandParent, String telemetryParent, Map<String, Variable> variableNamesAndTypesMap,
            Map<String, JavaFromXML.EnumValues> enumMap, JavaFromXML.ClassType classType, String salClassName) {
        this.className = computeClassName(className, classType);
        this.packageName = computePackageName(classType, basePackageName);
        this.superClassName = computeSuperClass(classType, stateParent, eventParent, commandParent, telemetryParent);
        this.enumerations = computeEnumerations(enumMap);
        this.variableNamesAndTypesMap = convertToTypeMap(variableNamesAndTypesMap);
        this.classType = classType;
        this.salClassName = salClassName;
        boolean isState = classType == JavaFromXML.ClassType.STATE;
        if (isState) {
            // TODO: This assumes one and only one enum, always to be called substate
            String type = className.replace("Event", "").replace("Detailed", "");
            String values = enumMap.values().iterator().next().getEnumConstants();
            this.enumerations.clear();
            final TypeSpec.Builder enumBuilder = TypeSpec.enumBuilder(type)
                    .addModifiers(Modifier.PUBLIC)
                    .addSuperinterface(SALEnum.class);
            for (String value : values.split(",")) {
                enumBuilder.addEnumConstant(value.trim());
            }
            this.enumerations.put("substate", enumBuilder.build());
            this.variableNamesAndTypesMap.clear();
            this.variableNamesAndTypesMap.put("substate", ClassName.bestGuess(type));
        }
        // Finally deal with inheritance.
        // TODO: This should not be hardwired here
        if (packageName.contains("atcamera") || packageName.contains("cccamera")) {
            String inherited = packageName.replace("atcamera", "camera");
            inherited = inherited.replace("cccamera", "camera");
            try {
                Class<?> inheritedClass = Class.forName(inherited + "." + this.className);
                // Check if constructor args match
                for (Constructor<?> constructor : inheritedClass.getConstructors()) {
                    final Map<String, TypeName> constructorArguments = getConstructorArguments();
                    List<TypeName> argTypes = new ArrayList(constructorArguments.values());
                    List<String> argNames = new ArrayList(constructorArguments.keySet());
                    if (constructor.getParameterCount() != constructorArguments.size()) {
                        continue;
                    }
                    boolean match = true;
                    for (int i = 0; i < constructor.getParameterCount(); i++) {
                        if (constructor.getParameterTypes()[i].isArray() != argTypes.get(i) instanceof ArrayTypeName) {
                            match = false;
                        } else if (!constructor.getParameters()[i].getName().equals(argNames.get(i))) {
                            match = false;
                        }
                    }
                    if (match) {
                        packageName = inherited;
                        this.isInherited = true;
                        variableNamesAndTypesMap.forEach((name, type) -> {
                            if (enumerations.containsKey(name)) {
                                TypeName typeName = ClassName.get(packageName + "." + this.className, JavaFromXML.capitalize(type.getType()));
                                if (type.isJavaArray()) {
                                    typeName = ArrayTypeName.of(typeName);
                                }
                                this.variableNamesAndTypesMap.put(name, typeName);
                            }
                        });
                        break;
                    }

                }
            } catch (ClassNotFoundException x) {
            }
        }
    }

    public String getClassName() {
        return className;
    }

    public Map<String, TypeName> getVariableNamesAndTypesMap() {
        return variableNamesAndTypesMap;
    }

    public String getPackageName() {
        return packageName;
    }

    public String getSuperClassName() {
        return superClassName;
    }

    public Set<TypeSpec> getEnumerations() {
        return new HashSet(enumerations.values());
    }

    JavaFromXML.ClassType getClassType() {
        return classType;
    }

    private static String computeClassName(String className, JavaFromXML.ClassType classType) {
        switch (classType) {
            case STATE:
            case EVENT:
                return className + "Event";
            case TELEMETRY:
                return className + "Telemetry";
            case COMMAND:
                return className + "Command";
            default:
                return className;
        }
    }

    private static String computePackageName(JavaFromXML.ClassType classType, String basePackageName) {
        switch (classType) {
            case STATE:
                return basePackageName + ".states";
            case EVENT:
                return basePackageName + ".event";
            case TELEMETRY:
                return basePackageName + ".telemetry";
            case COMMAND:
                return basePackageName + ".command";
            default:
                return basePackageName;
        }
    }

    private static String computeSuperClass(JavaFromXML.ClassType classType, String stateParent, String eventParent, String commandParent, String telemetryParent) {
        switch (classType) {
            case STATE:
                return stateParent;
            case EVENT:
                return eventParent;
            case TELEMETRY:
                return telemetryParent;
            case COMMAND:
                return commandParent;
            default:
                return commandParent;
        }
    }

    private Map<String, TypeName> convertToTypeMap(Map<String, Variable> variableNamesAndTypesMap) {
        Map<String, TypeName> result = new LinkedHashMap<>();
        variableNamesAndTypesMap.forEach((name, type) -> {
            result.put(name, mapVariableTypeToJava(type, name));
            variableNamesAndCounts.put(name, type);
        });
        return result;
    }

    private TypeName mapVariableTypeToJava(Variable variable, String name) {

        TypeName typeName;
        String type = variable.getType();
        if (type.equals("string")) {
            typeName = TypeName.get(String.class);
        } else if (type.equals("long long")) {
            typeName = TypeName.LONG;
        } else if (type.contains("long")) {
            typeName = TypeName.INT;
        } else if (enumerations.containsKey(name)) {
            typeName = ClassName.get(packageName + "." + className, JavaFromXML.capitalize(type));
        } else {
            typeName = TypeVariableName.get(type);
        }
        if (variable.isJavaArray()) {
            typeName = ArrayTypeName.of(typeName);
        }
        return typeName;
    }

    String getClassTypeName() {
        String whatClass = "";
        if (classType == JavaFromXML.ClassType.TELEMETRY) {
            whatClass = "telemetry";
        }
        if (classType == JavaFromXML.ClassType.COMMAND) {
            whatClass = "command";
        }
        if (classType == JavaFromXML.ClassType.EVENT) {
            whatClass = "event";
        }
        if (classType == JavaFromXML.ClassType.STATE) {
            whatClass = "state";
        }
        return whatClass;
    }

    String getSALClassName() {
        return salClassName;
    }

    Map<String, TypeName> getSuperClassArguments() {
        return superClassArguments;
    }

    Map<String, TypeName> getConstructorArguments() {
        Map<String, TypeName> map = new LinkedHashMap<>();
        map.putAll(superClassArguments);
        map.putAll(variableNamesAndTypesMap);
        return map;
    }

    private static Map<String, TypeSpec> computeEnumerations(Map<String, JavaFromXML.EnumValues> enumMap) {
        Map<String, TypeSpec> result = new LinkedHashMap<>();
        enumMap.forEach((name, values) -> {
            final TypeSpec.Builder enumerationBuilder = TypeSpec.enumBuilder(JavaFromXML.capitalize(values.getType()))
                    .addModifiers(Modifier.PUBLIC)
                    .addSuperinterface(SALEnum.class);
            for (String value : values.getEnumConstants().split(",")) {
                enumerationBuilder.addEnumConstant(value.trim());
            }
            result.put(name, enumerationBuilder.build());
        });
        return result;
    }

    boolean isEnumeration(String name) {
        return enumerations.containsKey(name);
    }

    boolean isArray(String name) {
        return variableNamesAndTypesMap.get(name) instanceof ArrayTypeName;
    }

    TypeSpec getEnumerationType(String name) {
        return enumerations.get(name);
    }

    int getCount(String name) {
        return variableNamesAndCounts.get(name).getCount();
    }

    int getSize(String name) {
        return variableNamesAndCounts.get(name).getSize();
    }

    String getGetter(String name) {
        TypeName type = variableNamesAndTypesMap.get(name);
        if (type == null) {
            type = TypeName.INT;
        }
        final String stem = type.toString().equalsIgnoreCase("boolean") ? "is" : "get";
        return stem + JavaFromXML.capitalize(name);
    }

    boolean isInherited() {
        return isInherited;
    }
}
