package org.lsst.sal.codegen;

import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;
import java.util.Map;

/**
 * Writes out the .sal file used at runtime to link the SAL classes and corresponding
 * SimpleSal classes.
 */
class SalWriter {

    private final List<ClassInfo> classInfos;
    private final String managerClassName;

    SalWriter(List<ClassInfo> classInfos, String managerClassName) {
        this.classInfos = classInfos;
        this.managerClassName = managerClassName;
    }

    void writeSalFile(String outputPath, String salFileName, String basePackageName) throws IOException {
        final File cameraSalDir = new File(outputPath + "/" + basePackageName.replace(".", "/") + "/");
        cameraSalDir.mkdirs();
        try (FileWriter cameraSalFile = new FileWriter(new File(cameraSalDir, salFileName));
            PrintWriter buffer = new PrintWriter(cameraSalFile)) {
            buffer.println("# Description of SAL interfaces");
            buffer.println("# Name of manager class");
            buffer.println(this.managerClassName);
            for (ClassInfo info : classInfos) {
                Map<String, TypeName> constructorVariables = info.getConstructorArguments();
                buffer.printf("%s %s.%s %s",info.getClassTypeName(),info.getPackageName(),info.getClassName(),info.getSALClassName());
                constructorVariables.forEach((name, typeName) -> {
                    if (info.isEnumeration(name)) {
                        if (info.isArray(name)) { typeName = ArrayTypeName.of(TypeName.INT); }
                        else { typeName = TypeName.INT; }
                    }
                    if ("substate".equals(name) && "SummaryStateEvent".equals(info.getClassName())) {
                        name = "summaryState";
                    }
                    if (typeName.toString().equals("java.lang.String[]")) {
                        typeName = ClassName.get(String.class);
                    }
                    buffer.printf(" %s %s", typeName.toString(), name);
                });
                buffer.println();
            }
        }
    }
}