package org.lsst.sal.codegen;

import com.squareup.javapoet.ArrayTypeName;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.lang.model.element.Modifier;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALCommandResponse;
import org.lsst.sal.SALEvent;
import org.lsst.sal.SALException;
import org.lsst.sal.SALReceivedCommand;
import org.lsst.sal.SALTelemetry;
import org.lsst.sal.codegen.JavaFromXML.ClassType;
import static org.lsst.sal.codegen.JavaFromXML.ClassType.COMMAND;

/**
 *
 * @author tonyj
 */
public class TestGenerator {

    private final String commandSendReceiveTestFile = "CommandSendReceiveTest";
    private final String eventSendReceiveTestFile = "EventSendReceiveTest";
    private final String telemetrySendReceiveTestFile = "TelemetrySendReceiveTest";
    private final Random random = new Random(12345);
    private final List<ClassInfo> classInfos;
    private final String basePackageName;
    private final String salFile;

    TestGenerator(List<ClassInfo> classInfos, String basePackageName, String salFile) {

        this.classInfos = classInfos;
        this.basePackageName = basePackageName;
        this.salFile = salFile;
    }

    void writeTestFiles(String testCodePath) throws IOException {
        generateBoilerPlate(testCodePath, ClassName.get(basePackageName, commandSendReceiveTestFile), JavaFromXML.ClassType.COMMAND);
        generateBoilerPlate(testCodePath, ClassName.get(basePackageName, eventSendReceiveTestFile), JavaFromXML.ClassType.EVENT);
        generateBoilerPlate(testCodePath, ClassName.get(basePackageName, telemetrySendReceiveTestFile), JavaFromXML.ClassType.TELEMETRY);
    }

    private void generateBoilerPlate(String outputPath, ClassName generatedClass, ClassType testType) throws IOException {

        List<MethodSpec> methods = new ArrayList<>();
        List<FieldSpec> fields = new ArrayList<>();

        TypeName sal = ClassName.get(SAL.class);
        final FieldSpec salField = FieldSpec.builder(sal, "sal", Modifier.STATIC, Modifier.PRIVATE).build();
        fields.add(salField);
        TypeName executor = ClassName.get(ExecutorService.class);
        final FieldSpec executorField = FieldSpec.builder(executor, "executor", Modifier.STATIC, Modifier.PRIVATE).build();
        fields.add(executorField);

        methods.add(createSetupMethod(executorField, salField));
        methods.add(createTeardownMethod(executorField, salField));
        final MethodSpec testSendReceive = createSendReceiveMethod(executorField, salField, testType);
        methods.add(testSendReceive);

        for (ClassInfo info : classInfos) {
            if (info.getClassType() == testType) {
                ClassName testClass = ClassName.get(info.getPackageName(), info.getClassName());

                final MethodSpec.Builder testCaseBuilder = MethodSpec.methodBuilder("sendReceive" + info.getClassName())
                        .addModifiers(Modifier.PUBLIC)
                        .addException(Exception.class)
                        .addAnnotation(Test.class);

                info.getConstructorArguments().forEach((name, type) -> {
                    Object testData = generateTestData(info, name, type);
                    if (testData.toString().contains("$3T")) {
                        testCaseBuilder.addStatement("$1T $2N = " + testData, type, name,
                                type instanceof ArrayTypeName ? ((ArrayTypeName) type).componentType : type);
                    } else {
                        testCaseBuilder.addStatement("$1T $2N = " + testData, type, name);
                    }

                });

                testCaseBuilder
                        .addStatement("$T $N = $N(new $T($N))", testSendReceive.returnType, "item", testSendReceive, testClass, String.join(",", info.getConstructorArguments().keySet()))
                        .addStatement("$T.assertTrue($N instanceof $T)", Assert.class, "item", testClass)
                        .addStatement("$T $N = ($T) $N", testClass, "item_", testClass, "item");

                info.getConstructorArguments().forEach((name, type) -> {
                    String assertEquals = "assertEquals";
                    if (type instanceof ArrayTypeName) {
                        assertEquals = "assertArrayEquals";
                        type = ((ArrayTypeName) type).componentType;
                    }
                    if ("float".equals(type.toString()) || "double".equals(type.toString())) {
                        testCaseBuilder.addStatement("$T.$N($N,$N.$N(),1e-6f)", Assert.class, assertEquals, name, "item_", info.getGetter(name));
                    } else {
                        testCaseBuilder.addStatement("$T.$N($N,$N.$N())", Assert.class, assertEquals, name, "item_", info.getGetter(name));
                    }
                });

                methods.add(testCaseBuilder.build());
            }

        }

        // Put together the whole class
        TypeSpec classDefinition = TypeSpec.classBuilder(generatedClass.simpleName())
                .addModifiers(Modifier.PUBLIC)
                .addFields(fields)
                .addMethods(methods)
                .build();

        JavaFile javaFile = JavaFile.builder(generatedClass.packageName(), classDefinition)
                .skipJavaLangImports(true)
                .build();

        javaFile.writeTo(new File(outputPath));
    }

    private Object generateTestData(ClassInfo info, String name, TypeName type) {
        String letters = "abcdefghijklmnopqrstuvwxyz";
        boolean isArray = type instanceof ArrayTypeName;
        if (isArray) {
            TypeName arrayType = ((ArrayTypeName) type).componentType;
            int arraySize = info.getCount(name);
            List<String> array = new ArrayList<>();
            for (int i = 0; i < arraySize; i++) {
                array.add(generateTestData(info, name, arrayType).toString());
            }
            return "{" + String.join(",", array) + "}";
        }
        switch (type.toString()) {
            case "short":
            case "int":
            case "long":
                return random.nextInt(1000);
            case "double":
            case "float":
                return random.nextFloat() + "f";
            case "boolean":
                return random.nextBoolean();
            case "java.lang.String":
                int n = random.nextInt(info.getSize(name) + 1);
                StringBuilder sb = new StringBuilder();
                for (int i = 0; i < n; i++) {
                    int l = random.nextInt(letters.length());
                    sb.append(letters.substring(l, l + 1));
                }
                return "\"" + sb + "\"";
            default:
                if (info.isEnumeration(name)) {
                    TypeSpec typeSpec = info.getEnumerationType(name);
                    int enumSize = typeSpec.enumConstants.size();
                    int enumSelected = random.nextInt(enumSize);
                    return "$3T." + new ArrayList(typeSpec.enumConstants.keySet()).get(enumSelected);
                }
                return "null";
        }
    }

    private static MethodSpec createSendReceiveMethod(final FieldSpec executorField, final FieldSpec salField, final ClassType testType) {
        Class testClass;
        String getterMethodName;
        Class getterReturnType;
        String sendMethodName;
        switch (testType) {
            case COMMAND:
                testClass = SALCommand.class;
                getterMethodName = "getNextCommand";
                getterReturnType = SALReceivedCommand.class;
                sendMethodName = "issueCommand";
                break;
            case TELEMETRY:
                testClass = SALTelemetry.class;
                getterMethodName = "getTelemetry";
                getterReturnType = SALTelemetry.class;
                sendMethodName = "sendTelemetry";
                break;
            default:
                testClass = SALEvent.class;
                getterMethodName = "getNextEvent";
                getterReturnType = SALEvent.class;
                sendMethodName = "logEvent";
        }
        final MethodSpec.Builder methodBuilder = MethodSpec.methodBuilder("testSendReceive")
                .addModifiers(Modifier.PRIVATE)
                .addParameter(testClass, "item")
                .returns(testClass)
                .addException(InterruptedException.class)
                .addException(SALException.class)
                .addException(ExecutionException.class)
                .addException(TimeoutException.class)
                .addStatement("$T $N = $N.submit(() -> $N.$N($T.ofSeconds(10)))", ParameterizedTypeName.get(Future.class, getterReturnType), "future", executorField, "sal", getterMethodName, Duration.class)
                .beginControlFlow("try");

        if (testType == COMMAND) {
            methodBuilder
                    .addStatement("$T $N = $N.$N($N)", SALCommandResponse.class, "response", salField, sendMethodName, "item")
                    .addStatement("$T $N = $N.get(10, $T.SECONDS)", SALReceivedCommand.class, "result", "future", TimeUnit.class)
                    .addStatement("$N.acknowledgeCommand($T.ofSeconds(3))", "result", Duration.class)
                    .addStatement("$N.reportComplete()", "result")
                    .addStatement("int rc = $N.waitForCompletion($T.ofSeconds(1))", "response", Duration.class)
                    .addStatement("$T.assertEquals(303,rc)", Assert.class)
                    .addStatement("return $N.getCommand()", "result");
        } else {
            methodBuilder
                    .addStatement("$N.$N($N)", "sal", sendMethodName, "item")
                    .addStatement("return $N.get(10, $T.SECONDS)", "future", TimeUnit.class);
        }

        return methodBuilder
                .nextControlFlow("finally")
                .addStatement("$N.cancel(true)", "future")
                .endControlFlow()
                .build();
    }

    private MethodSpec createTeardownMethod(final FieldSpec executorField, final FieldSpec salField) {
        return MethodSpec.methodBuilder("tearDownClass")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addException(InterruptedException.class)
                .addException(SALException.class)
                .addAnnotation(AfterClass.class)
                .addStatement("$N.shutdown()", executorField)
                .addStatement("$N.awaitTermination(10, $T.SECONDS)", executorField, TimeUnit.class)
                .addStatement("$N.close()", salField)
                .build();
    }

    private MethodSpec createSetupMethod(final FieldSpec executorField, final FieldSpec salField) {
        return MethodSpec.methodBuilder("setUpClass")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addAnnotation(BeforeClass.class)
                .addStatement("$N = $N.create()", salField, this.salFile)
                .addStatement("$N = $T.newFixedThreadPool(1)", executorField, ClassName.get(Executors.class))
                .build();
    }
}
