package org.lsst.sal.codegen;

import java.io.File;
import java.util.Map;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.w3c.dom.Element;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;

/**
 *
 * @author Farrukh Azfar, generate JavaCode for SALTelemetry, CameraStateEvent
 * and SALEvent classes - generate: class, constructor, variable declarations,
 * setter getter functions and toString() methods - many thanks to Max Turri for
 * helping with the interface
 *
 * Usage : java -jar ./target/org-lsst-sal-camera-1.5.1-SNAPSHOT.jar
 * /home/azfarl/java_xml_java/atcamera_Events.xml /home/azfarl/codegen/ USES JAR
 * file from path: Netbeans/rep-direcoryname/org-lsst-camera-simple-sal ^XML
 * File ^Output for the many java files
 */
public class JavaFromXML {

    private final String basePackageName;
    private final String outputPath;
    private final File[] xmlFilelist;
    private final String managerClassName;
    private final String stateParent;
    private final String eventParent;
    private final String commandParent;
    private final String telemetryParent;
    private final String salFileName;

    static class EnumValues {

        private final String type;
        private final String values;

        public EnumValues(String type, String values) {
            this.type = type;
            this.values = values;
        }

        public String getType() {
            return type;
        }

        public String getEnumConstants() {
            return values;
        }
    }

    static class Variable {

        private final int count;
        private final int size;
        private final String type;

        public Variable(String type, int count, int size) {
            this.type = type;
            this.count = count;
            this.size = size;
        }

        public int getCount() {
            return count;
        }

        public String getType() {
            return type;
        }

        public int getSize() {
            return size;
        }

    }

    enum ClassType {
        TELEMETRY("SALTelemetry"), STATE("SALEvent"), EVENT("SALEvent"), COMMAND("SALCommand");
        private final String tag;

        ClassType(String tag) {
            this.tag = tag;
        }

        public String getTag() {
            return tag;
        }

    };

    public JavaFromXML(String basePackageName, String managerClassName, String salFileName, String stateParent, String eventParent, String commandParent, String telemetryParent, String salMainJava, String outputPath, String testOutputPath, String outputResoucesPath, File[] xmlFileList) throws ParserConfigurationException, SAXException, IOException {

        this.basePackageName = basePackageName;
        this.outputPath = outputPath;
        this.salFileName = salFileName;
        this.xmlFilelist = xmlFileList;
        this.managerClassName = managerClassName;
        this.stateParent = stateParent;
        this.eventParent = eventParent;
        this.commandParent = commandParent;
        this.telemetryParent = telemetryParent;

        // first string className,
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();

        ArrayList<ClassInfo> classInfos = new ArrayList<>();
        for (File xmlFile : xmlFileList) {

            ReadXMLFillArray xmlContent = new ReadXMLFillArray(builder, xmlFile.getAbsolutePath());
            classInfos.addAll(xmlContent.getClassInfos());
        }

        // write classes
        for (ClassInfo info : classInfos) {
            if (!info.isInherited()) {
                JavaClassWriter writer = new JavaClassWriter(info);
                writer.writeClasses(outputPath);
            }
        }

        //write camera.sal - could it be named something else - 
        SalWriter salWriter = new SalWriter(classInfos, managerClassName);
        salWriter.writeSalFile(outputResoucesPath, salFileName, basePackageName);

        TestGenerator testGenerator = new TestGenerator(classInfos, basePackageName, salMainJava);
        testGenerator.writeTestFiles(testOutputPath);

    }

    private class ReadXMLFillArray {

        private List<ClassInfo> classInfoCollection = new ArrayList<>();
        private boolean stateEvent;
        private String rawClassName;

        public ReadXMLFillArray(DocumentBuilder builder, String xmlFileName) throws SAXException {

            try {
                // read the array - generate class names
                Document document = builder.parse(new InputSource(new FileReader(xmlFileName)));

                // now start reading
                Element root = document.getDocumentElement();
                System.out.println("ROOT ELEMENT : " + root.getTagName());

                if ("SALObjects".equals(root.getTagName())) {
                    NodeList childNodes = root.getChildNodes();
                    for (int i = 0; i < childNodes.getLength(); i++) {
                        Node item = childNodes.item(i);
                        if (item instanceof Element) {
                            Element element = (Element) item;
                            // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
                            ClassType classType;
                            if (element.getTagName().contains("Event")) {
                                classType = ClassType.EVENT; // caution this has two subtypes - Event and Telemetry
                            } else if (element.getTagName().contains("Telemetry")) {
                                classType = ClassType.TELEMETRY;
                            } else {
                                classType = ClassType.COMMAND;
                            }
                            System.out.println("***Handling classType " + classType);
                            handleObjectsOfClass(element, classType);
                        }
                    }
                } else {

                    // class type : ie Event or Telemetry is set here by grabbing the requisite text from root -
                    ClassType classType;
                    if (root.getTagName().contains("Event")) {
                        classType = ClassType.EVENT; // caution this has two subtypes - Event and State
                    } else if (root.getTagName().contains("Telemetry")) {
                        classType = ClassType.TELEMETRY;
                    } else {
                        classType = ClassType.COMMAND;
                    }

                    handleObjectsOfClass(root, classType);
                }

            } catch (IOException ex) {
                System.out.println("Exception caught in readXMLFileArray: \n");
                Logger.getLogger(JavaFromXML.class.getName()).log(Level.SEVERE, null, ex);
            }

        }

        private void handleObjectsOfClass(Element document, ClassType classType) throws NumberFormatException, DOMException {
            // get the actual class blocks...
            NodeList salClassNodes = document.getElementsByTagName(classType.getTag());
            // now looping over each class -

            String className;

            // loop over all classes -
            for (int i = 0; i < salClassNodes.getLength(); i++) {

                // for each class you have one of these
                Map<String, Variable> variableNamesAndTypesMap = new LinkedHashMap<>();
                Map<String, EnumValues> enumMap = new LinkedHashMap<>();
                // Need another list to keep track of what is actually stored in the ENUM for SAL particularly camera.sal class
//                Map<String, String> enumTypeMapSAL = new LinkedHashMap<>();
//                Map<String, String> enumStateTypeMapSAL = new LinkedHashMap<>();
                // now find all EFDB Topics as these define the class name
                NodeList salClassContentNodes = salClassNodes.item(i).getChildNodes();

                // salNodes will be read in - even blank spaces will be read in so we
                // only want to pick those that are bonafide elements
                if (salClassNodes.item(i) instanceof Element) {

                    Element eClass = (Element) salClassNodes.item(i);

                    // find a child -
                    NodeList topicList = eClass.getElementsByTagName("EFDB_Topic");
                    // should only be ONE topic (ie className) per class
                    if (topicList.getLength() > 1) {
                        System.out.println("More than one EFDB_Topic(Class Name) in class - incorrect xml \n");
                        System.exit(0);
                    }

                    className = topicList.item(0).getTextContent().trim();
                    rawClassName = className.split("_", 2)[0] + "." + className.split("_", 2)[1];
                    // split it up grab the last bit which is the class name
                    className = className.split("_", 2)[1];
                    className = className.replace("logevent_","");

                    if (className.contains("ccs")) {
                        className = className.replace("ccs", "CCS");
                    } else {
                        // otherwise just capitalize className's first character -
                        className = className.substring(0, 1).toUpperCase() + className.substring(1);
                    }
                    System.out.printf("+++++++ %s %s\n", rawClassName, className);
                    // State Events are treated differently
                    ClassType thisClassType = classType;
                    if (className.endsWith("State")) {
                        thisClassType = ClassType.STATE;
                    }

                    // find the list of items ...
                    NodeList itemList = eClass.getElementsByTagName("item");
                    for (int iI = 0; iI < itemList.getLength(); iI++) {

                        if (itemList.item(iI) instanceof Element) {

                            // fine obly real items
                            Element eI = (Element) itemList.item(iI);

                            // there should only ever be ONE per item of the following
                            NodeList efdbNameList = eI.getElementsByTagName("EFDB_Name");
                            NodeList idlTypeList = eI.getElementsByTagName("IDL_Type");
                            NodeList idlSizeList = eI.getElementsByTagName("IDL_Size");
                            NodeList enumerationList = eI.getElementsByTagName("Enumeration");
                            NodeList countList = eI.getElementsByTagName("Count");

                            // bail out if someone has given us an incorrect xml ....
                            if ((efdbNameList.getLength() > 1 || idlTypeList.getLength() > 1 || enumerationList.getLength() > 1)
                                    || (efdbNameList.getLength() != idlTypeList.getLength())) {
                                System.out.println(" There should only be 1 EFDB_Name, IDL_Type, Enumeration List in an item \n ");
                                System.out.println(" Every EFDB_Name should have a corresponding IDL_Type");
                                System.exit(0);
                            }
                            int count = Integer.parseInt(countList.item(0).getTextContent().trim());
                            int size = 0;
                            if (idlSizeList.getLength() >= 1) {
                                size = Integer.parseInt(idlSizeList.item(0).getTextContent().trim());
                            }
                            // summary state fix 
                            if (className.endsWith("SummaryState")) {
                                enumMap.put(efdbNameList.item(0).getTextContent().trim(), new EnumValues("SummaryState", "DISABLED,ENABLED,FAULT,OFFLINE,STANDBY"));
                            }
                            if (enumerationList.getLength() > 0) {
                                String enumType = efdbNameList.item(0).getTextContent().trim();

                                enumMap.put(efdbNameList.item(0).getTextContent().trim(), new EnumValues(enumType, enumerationList.item(0).getTextContent().trim()));
                                variableNamesAndTypesMap.put(efdbNameList.item(0).getTextContent().trim(), new Variable(enumType, count, size));
                            } else if (efdbNameList.getLength() > 0 && enumerationList.getLength() < 1) {
                                String type = idlTypeList.item(0).getTextContent().trim();

                                //silly fix for value boolean - need to exclude it from command files until xml is fixed - Farrukh
                                if (classType.equals(ClassType.COMMAND) && efdbNameList.item(0).getTextContent().trim().equals("value") && type.equals("boolean")) {
                                    //do nothing
                                    System.out.println("boolean variable value found for Command Class - thrown out");
                                } else {

                                    variableNamesAndTypesMap.put(efdbNameList.item(0).getTextContent().trim(), new Variable(type, count, size));
                                }
                            }
                        }

                    }

                    this.classInfoCollection.add(new ClassInfo(rawClassName, className, basePackageName, stateParent, eventParent, commandParent, telemetryParent, variableNamesAndTypesMap, enumMap, thisClassType, rawClassName));
                }
            } // end of loop over SAL classes
        }

        /**
         *
         * @return
         */
        public List<ClassInfo> getClassInfos() {

            return this.classInfoCollection;

        }
    }

    static String capitalize(String in) {
        return in.substring(0, 1).toUpperCase() + in.substring(1);
    }

}
