package org.lsst.sal.codegen;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.Modifier;

/**
 * This class now uses JavaPoet for code generation.
 *
 * @see <a href="https://github.com/square/javapoet">JavaPoet</a>
 * @author farrukh
 */
class JavaClassWriter {

    private final ClassInfo classInfo;

    JavaClassWriter(ClassInfo classInfo) {
        this.classInfo = classInfo;
    }

    // master for writing class
    void writeClasses(String outputPath) throws IOException {

        List<FieldSpec> fields = new ArrayList<>();
     
        // Generate fields
        classInfo.getVariableNamesAndTypesMap().forEach((name,type) -> {
            fields.add(FieldSpec.builder(type, name, Modifier.PRIVATE, Modifier.FINAL).build());
        });

        List<MethodSpec> methods = new ArrayList<>();

        methods.add(constructConstructor());

        // Create getters
        classInfo.getVariableNamesAndTypesMap().forEach((name, type) -> {
            methods.add(constructGetter(name, type));
        });
        
        methods.add(constructToStringMethod());
                
        // Put together the whole class
        TypeSpec classDefinition = TypeSpec.classBuilder(classInfo.getClassName())
                .addModifiers(Modifier.PUBLIC)
                .superclass(ClassName.bestGuess(classInfo.getSuperClassName()))
                .addMethods(methods)
                .addFields(fields)
                .addTypes(classInfo.getEnumerations())
                .build();

        // Write it out
        JavaFile javaFile = JavaFile.builder(classInfo.getPackageName(), classDefinition)
                .skipJavaLangImports(true)
                .build();

        javaFile.writeTo(new File(outputPath));
    }
    
    private MethodSpec constructGetter(String name, TypeName type) {
        final MethodSpec.Builder getterBuilder = MethodSpec.methodBuilder(classInfo.getGetter(name))
                .addModifiers(Modifier.PUBLIC)
                .returns(type)
                .addStatement("return $L", name);
        if (name.equals("substate")) {
            getterBuilder.addAnnotation(Override.class);
        }
        return getterBuilder.build();
    }

    private MethodSpec constructToStringMethod() {
        final CodeBlock.Builder toStringCodeBuilder = CodeBlock.builder()
                .add("return $S + ", classInfo.getClassName() + "{");

        String delimiter = "";
        for (Map.Entry<String, TypeName> var : classInfo.getVariableNamesAndTypesMap().entrySet()) {
            String name = var.getKey();
            toStringCodeBuilder.add("$S + $N +$Z", delimiter + name + "=", name);
            delimiter = ", ";
        }

        CodeBlock toStringCode = toStringCodeBuilder
                .add("$S","}")
                .build();

        return MethodSpec.methodBuilder("toString")
                .addModifiers(Modifier.PUBLIC)
                .addAnnotation(Override.class)
                .returns(String.class)
                .addStatement(toStringCode)
                .build();
    }

    private MethodSpec constructConstructor() {
        List<ParameterSpec> parameters = new ArrayList<>();

        // Generate constructor parameters
        classInfo.getConstructorArguments().forEach((name,type) -> {
            parameters.add(ParameterSpec.builder(type, name).build());
        });

        // Generate constructor
        final MethodSpec.Builder constructorBuilder = MethodSpec.constructorBuilder()
                .addModifiers(Modifier.PUBLIC)
                .addParameters(parameters);

        if (!classInfo.getSuperClassArguments().isEmpty()) {
            constructorBuilder.addStatement("super ($N)",String.join(",", classInfo.getSuperClassArguments().keySet()));
        }

        classInfo.getVariableNamesAndTypesMap().entrySet().forEach((var) -> {
            String name = var.getKey();
            constructorBuilder.addStatement("this.$1L = $1L", name);
        });
        return constructorBuilder.build();
    }
}
