package org.lsst.sal;

import DDS.ANY_INSTANCE_STATE;
import DDS.ANY_SAMPLE_STATE;
import DDS.ANY_VIEW_STATE;
import DDS.SampleInfoSeqHolder;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.concurrent.TimeoutException;
import org.lsst.sal.SALCommandResponse.CommandFailedException;

/**
 * A temporary class to make up for inadequacies in the Java implementation of
 * SAL with respect to command responses. Should be replaced once SAL is fixed.
 * 
 * @see <a href="https://jira.lsstcorp.org/projects/TSSAL/issues/TSSAL-19">TSSAL-19</a>
 * @author tonyj
 */
class CommandHelper {

    private final Object manager;
    private final Class ackcmdSeqHolder;
    private final int debugLevel = 0;
    private final Method getReader2Method;
    private final int cmdInProgress;
    private final int cmdComplete;
    private final int cmdNoAck;
    private final int cmdAckcmdActor;
    private final Field valueField;
    private final Class<?> valueType;
    private final Field ackField;
    private final Field errorField;
    private final Field resultField;
    private Field seqnumField;

    CommandHelper(Object manager) throws ReflectiveOperationException {
        this.manager = manager;
        final Class managerClass = manager.getClass();
        /// Kludgy way of find class of ackcmdSeqHolder
        Method[] methods = managerClass.getMethods();
        Class localAckcmdSeqHolder = null;
        for (Method method : methods) {
            if (method.getName().startsWith("getResponse")) {
                localAckcmdSeqHolder = method.getParameterTypes()[0];
                break;
            }
        }
        if (localAckcmdSeqHolder == null) {
            throw new NoSuchMethodException("getResponse");
        }
        this.ackcmdSeqHolder = localAckcmdSeqHolder;
        getReader2Method = managerClass.getMethod("getReader2", Integer.TYPE);
        cmdInProgress = managerClass.getField("SAL__CMD_INPROGRESS").getInt(manager);
        cmdComplete = managerClass.getField("SAL__CMD_COMPLETE").getInt(manager);
        cmdNoAck = managerClass.getField("SAL__CMD_NOACK").getInt(manager);
 
        int localcmdAckcmdActor = -1;
        for (Field field : managerClass.getFields()) {
            if (field.getName().endsWith("ackcmd_ACTOR")) {
                localcmdAckcmdActor = field.getInt(manager);
                break;
            }
        }
        if (localcmdAckcmdActor == -1) {
            throw new NoSuchFieldException("ackcmd_ACTOR");
        }
        cmdAckcmdActor = localcmdAckcmdActor;

        valueField = ackcmdSeqHolder.getField("value");
        valueType = valueField.getType().getComponentType();
        ackField = valueType.getField("ack");
        errorField = valueType.getField("error");
        resultField = valueType.getField("result");
        seqnumField = valueType.getField("private_seqNum");
    }

    @SuppressWarnings("SleepWhileInLoop")
    int waitForCompletion(int cmdId, int timeout) throws SALException, TimeoutException {
        try {
            Object ackcmd = ackcmdSeqHolder.newInstance();
            UsefulResponse useful = new UsefulResponse();
            long timeoutTime = System.currentTimeMillis()+timeout*1000;
            while (true) {
                int status = getResponseInternal(cmdId, ackcmd, useful);
                if (status == cmdComplete) {
                    return status;
                }
                if (timeoutTime<System.currentTimeMillis()) break;
                Thread.sleep(10);
            }
            throw new java.util.concurrent.TimeoutException("waitForCompletion timed out");
        } catch (InterruptedException | ReflectiveOperationException x) {
            throw new SALException("Error waiting for command response", x);
        }
    }
    
    @SuppressWarnings("SleepWhileInLoop")
    Duration waitForAck(int cmdId, int timeout) throws CommandFailedException, TimeoutException, SALException {
        try {
            Object ackcmd = ackcmdSeqHolder.newInstance();
            UsefulResponse useful = new UsefulResponse();
            long timeoutTime = System.currentTimeMillis()+timeout*1000;
            while (true) {
                int status = getResponseInternal(cmdId, ackcmd, useful);
                if (status == cmdInProgress) {
                    return Duration.ofSeconds(useful.error);
                } else if (status == cmdComplete) {
                    return Duration.ZERO;
                }
                if (timeoutTime<System.currentTimeMillis()) break;
                Thread.sleep(10);
            }
            throw new java.util.concurrent.TimeoutException("waitForAck timed out");
        } catch (InterruptedException | ReflectiveOperationException x) {
            throw new SALException("Error waiting for command ack", x);
        }
    }

    private int getResponseInternal(int cmdId, Object data, UsefulResponse useful) throws ReflectiveOperationException, CommandFailedException {

        Object dreader = getReader2Method.invoke(manager, cmdAckcmdActor);
        Method takeMethod = dreader.getClass().getMethod("take", data.getClass(), SampleInfoSeqHolder.class, Integer.TYPE, Integer.TYPE, Integer.TYPE, Integer.TYPE);
        Method returnMethod = dreader.getClass().getMethod("return_loan", data.getClass(), SampleInfoSeqHolder.class);
        SampleInfoSeqHolder infoSeq = new SampleInfoSeqHolder();

        takeMethod.invoke(dreader, data, infoSeq, 1,
                ANY_SAMPLE_STATE.value,
                ANY_VIEW_STATE.value,
                ANY_INSTANCE_STATE.value);
        try {
            Object value = valueField.get(data);

            if (value != null && Array.getLength(value) > 0) {
                int status = cmdNoAck;
                for (int i = 0; i < Array.getLength(value); i++) {
                    Object v = Array.get(value, i);
                    int seqnum = seqnumField.getInt(v);
                    if (seqnum != cmdId) {
                        continue;
                    }
                    useful.ack =  ackField.getInt(v);
                    useful.error = errorField.getInt(v);
                    useful.result = (String) resultField.get(v);
                    if (debugLevel > 8) {
                        System.out.println("=== [getResponse_sequence] message received :");
                        System.out.println("    ack      : " + useful.ack);
                        System.out.println("    error    : " + useful.error);
                        System.out.println("    result   : " + useful.result);
                        System.out.println("    seqnum   : " + seqnum);
                    }
                    status = useful.ack;
                    if (status < 0) {
                        throw new CommandFailedException(useful.result, useful.ack, useful.error);
                    } 
                }
                return status; 
            } else {
                if (debugLevel > 8) {
                    System.out.println("=== [getResponse_sequence] No ack yet!");
                }
                return cmdNoAck;
            }
        } finally {
            returnMethod.invoke(dreader, data, infoSeq);
        }
    }

    private static class UsefulResponse {

        private int ack;
        private int error;
        private String result;
    }
}
