package org.lsst.sal;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeoutException;

/**
 * An implementation of the SAL interface which works using reflection and a
 * description of the required interfaces using a .sal file. The .sal file
 * contains three types of line:
 * <ul>
 * <li>Lines beginning with #, which are treated as comments</li>
 * <li>The first non-comment line, which is taken to be the name of the
 * salmanager class</li>
 * <li>All other lines, which are taken to be descriptions of SAL objects of the
 * form:
 * <pre>
 * &lt;type&gt; &lt;imple-SAL class name&gt; &lt;SAL class name&gt; [&lt;argument list&gt;]
 * </pre>
 * </li>
 * </ul>
 *
 * @author tonyj
 * @param <C> The base class for all commands sent and received by this
 * implementation
 * @param <E> The base class for all events sent and received by this
 * implementation
 * @param <T> The base class for all telemetry sent and received by this
 * implementation
 */
public class SALImplementation<C extends SALCommand, E extends SALEvent, T extends SALTelemetry> implements SAL<C, E, T> {

    private Object manager;
    private Method salProcessor;
    private Method salEvent;
    private int cmdInProgress;
    private int cmdComplete;
    private int cmdFailed;
    private int cmdNoPerm;
    private Method shutdownMethod;
    private int salOK;
    private Method salTelemetrySub;
    private Method salTelemetryPub;
    private Method salEventSub;
    private Method salEventPub;
    private CommandHelper commandHelper;

    private enum SALType {
        COMMAND, EVENT, STATE, TELEMETRY
    };
    private final List<SALCommandItem> commands = new ArrayList<>();
    private final List<SALEventObject> events = new ArrayList<>();
    private final List<SALTelemetryObject> telemetry = new ArrayList<>();

    /**
     * Convenience method for reading configuration from a resource
     *
     * @param c The class to use to access the resource
     * @param resource The name of the resource
     */
    protected SALImplementation(Class c, String resource) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(c.getResourceAsStream(resource)))) {
            init(reader);
        } catch (IOException | NullPointerException x) {
            throw new RuntimeException("Error reading resource: " + resource, x);
        }
    }

    /**
     * Create a GenericSAL object by reading a .sal file.
     *
     * @param salFile A reader from which the file will be read.
     * @throws IOException If an IO or syntax error occurs.
     */
    public SALImplementation(BufferedReader salFile) throws IOException {
        init(salFile);
    }

    private void init(BufferedReader salFile) throws IOException, SecurityException, IllegalArgumentException {
        boolean firstLine = true;
        for (int n = 1;; n++) {
            String line = salFile.readLine();
            try {
                if (line == null) {
                    break;
                }
                line = line.trim();
                // TODO: We should allow # to occur after valid line
                if (line.startsWith("#") || line.length() == 0) {
                    continue;
                }
                if (firstLine) {
                    createSALManager(line);
                    firstLine = false;
                } else {
                    String[] split = line.split("\\s+");
                    if (split.length < 3 || (split.length - 3) % 2 != 0) {
                        throw new IOException("Bad line: " + line);
                    }
                    createSALObject(split);
                }
            } catch (IOException x) {
                throw new IOException(String.format("Error reading line %d: %s", n, line), x);
            }
        }
    }

    private void createSALManager(String line) throws IllegalArgumentException, SecurityException, IOException {
        try {
            Class managerClass = Class.forName(line);
            manager = managerClass.newInstance();
            shutdownMethod = managerClass.getMethod("salShutdown");
            cmdInProgress = managerClass.getField("SAL__CMD_INPROGRESS").getInt(manager);
            cmdComplete = managerClass.getField("SAL__CMD_COMPLETE").getInt(manager);
            cmdFailed = managerClass.getField("SAL__CMD_FAILED").getInt(manager);
            cmdNoPerm = managerClass.getField("SAL__CMD_NOPERM").getInt(manager);
            salOK = managerClass.getField("SAL__OK").getInt(manager);
            salProcessor = managerClass.getMethod("salProcessor", String.class);
            salEvent = managerClass.getMethod("salEvent", String.class);
            salEventSub = managerClass.getMethod("salEventSub", String.class);
            salEventPub = managerClass.getMethod("salEventPub", String.class);
            salTelemetrySub = managerClass.getMethod("salTelemetrySub", String.class);
            salTelemetryPub = managerClass.getMethod("salTelemetryPub", String.class);
            commandHelper = new CommandHelper(manager);
        } catch (ReflectiveOperationException x) {
            throw new IOException("Error initializing manager class: " + line, x);
        }
    }

    private void createSALObject(String[] line) throws IOException {
        try {
            SALType type = SALType.valueOf(line[0].toUpperCase());
            switch (type) {
                case COMMAND:
                    commands.add(new SALCommandItem(line));
                    break;
                case EVENT:
                case STATE:
                    events.add(new SALEventObject(line));
                    break;
                case TELEMETRY:
                    telemetry.add(new SALTelemetryObject(line));
                    break;
                default:
                    throw new IOException("Unsupported type " + type);
            }

        } catch (ReflectiveOperationException ex) {
            throw new IOException("Error parsing SALFile", ex);
        }
    }

    @Override
    @SuppressWarnings("SleepWhileInLoop")
    public SALReceivedCommand<C> getNextCommand(Duration timeout) throws SALException {
        return poll(commands, timeout);
    }

    @Override
    public SALCommandResponse issueCommand(C command) throws SALException {
        return send(commands, command);
    }

    @Override
    public void logEvent(E event) throws SALException {
        send(events, event);
    }

    @Override
    @SuppressWarnings("SleepWhileInLoop")
    public E getNextEvent(Duration timeout) throws SALException {
        return poll(events, timeout);
    }

    @Override
    public void sendTelemetry(T telem) throws SALException {
        send(telemetry, telem);
    }

    @Override
    public T getTelemetry(Duration timeout) throws SALException {
        return poll(telemetry, timeout);
    }

    @SuppressWarnings("SleepWhileInLoop")
    private <T extends SALObject, R> R poll(List<T> objects, Duration timeout) throws SALException {
        Instant stop = Instant.now().plus(timeout);

        while (!Instant.now().isAfter(stop)) {

            // Currently we have to poll for each item
            for (T item : objects) {
                R result = (R) item.pollForObject();
                if (result != null) {
                    return result;
                }
            }
            try {
                // FIXME: Would be great if we did not have to poll
                Thread.sleep(10);
            } catch (InterruptedException ex) {
                throw new SALException("Unexpected interupt while polling for object", ex);
            }
        }
        return null; // Timeout  
    }

    private <T extends SALObject, SS, R> R send(List<T> objects, SS message) throws SALException {
        for (T salItem : objects) {
            if (salItem.getSimpleSALClass().isAssignableFrom(message.getClass())) {
                return (R) salItem.send(message);
            }
        }
        throw new SALException("Object cannot be sent: "+message);
    }

    @Override
    public void close() throws SALException {
        try {
            shutdownMethod.invoke(manager);
        } catch (ReflectiveOperationException x) {
            throw new SALException("Error shuting down manager", x);
        }
    }

    private class SALTelemetryObject<SS extends SALTelemetry, S> extends SALObject<SS, S> {

        private Method putSampleMethod;
        private Method getSampleMethod;

        SALTelemetryObject(String[] line) throws ReflectiveOperationException, IOException {
            super(line);
            if (!SALTelemetry.class.isAssignableFrom(getSimpleSALClass())) {
                throw new IOException("Class " + getSimpleSALClass() + " not valid for type " + getType());
            }
            salTelemetrySub.invoke(manager, line[2].replace(".", "_"));
            salTelemetryPub.invoke(manager, line[2].replace(".", "_"));
            putSampleMethod = manager.getClass().getMethod("putSample", getSalClass());
            getSampleMethod = manager.getClass().getMethod("getSample", getSalClass());
        }

        @Override
        SS pollForObject() throws SALException {
            try {
                S salInstance = getSALInstance();
                int rc = (int) getSampleMethod.invoke(manager, salInstance);
                if (rc == salOK) {
                    return convertFromSAL(salInstance);
                } else {
                    return null;
                }
            } catch (ReflectiveOperationException ex) {
                throw new SALException("Error invoking logEvent method", ex);
            }
        }

        @Override
        Object send(SS message) throws SALException {
            try {
                putSampleMethod.invoke(manager, convertToSAL(message));
            } catch (ReflectiveOperationException ex) {
                throw new SALException("Error invoking putSample method", ex);
            }
            return null;
        }

    }

    private class SALEventObject<SS extends SALEvent, S> extends SALObject<SS, S> {

        private Method logEventMethod;
        private Method getEventMethod;

        SALEventObject(String[] line) throws ReflectiveOperationException, IOException {
            super(line);
            if (!SALEvent.class.isAssignableFrom(getSimpleSALClass())) {
                throw new IOException("Class " + getSimpleSALClass() + " not valid for type " + getType());
            }
            final String eventAlias = line[2].replace(".", "_");
            salEvent.invoke(manager, eventAlias);
            salEventPub.invoke(manager, eventAlias);
            salEventSub.invoke(manager, eventAlias);

            logEventMethod = manager.getClass().getMethod("logEvent_" + line[2].replaceFirst(".*logevent_", ""), getSalClass(), Integer.TYPE);
            getEventMethod = manager.getClass().getMethod("getEvent_" + line[2].replaceFirst(".*logevent_", ""), getSalClass());
        }

        @Override
        SS pollForObject() throws SALException {
            try {
                S salInstance = getSALInstance();
                int rc = (int) getEventMethod.invoke(manager, salInstance);
                if (rc == salOK) {
                    return convertFromSAL(salInstance);
                } else {
                    return null;
                }
            } catch (ReflectiveOperationException ex) {
                throw new SALException("Error invoking logEvent method", ex);
            }
        }

        @Override
        Object send(SS message) throws SALException {
            try {
                logEventMethod.invoke(manager, convertToSAL(message), message.getPriority());
                return null;
            } catch (ReflectiveOperationException ex) {
                throw new SALException("Error invoking logEvent method", ex);
            }
        }
    }

    private class SALCommandItem<SS extends SALCommand, S> extends SALObject<SS, S> {

        private Method acceptMethod;
        private Method acknowledgeMethod;
        private Method issueMethod;

        SALCommandItem(String[] line) throws ReflectiveOperationException, IOException {
            super(line);
            if (!SALCommand.class.isAssignableFrom(getSimpleSALClass())) {
                throw new IOException("Class " + getSimpleSALClass() + " not valid for type " + getType());
            }
            salProcessor.invoke(manager, line[2].replace(".", "_"));
            acceptMethod = manager.getClass().getMethod("acceptCommand_" + line[2].replaceFirst(".*command_", ""), getSalClass());
            acknowledgeMethod = manager.getClass().getMethod("ackCommand_" + line[2].replaceFirst(".*command_", ""), Integer.TYPE, Integer.TYPE, Integer.TYPE, String.class);
            issueMethod = manager.getClass().getMethod("issueCommand_" + line[2].replaceFirst(".*command_", ""), getSalClass());
        }

        private void acknowledge(int cmdId, S response, int timeout, String message) throws SALException {
            try {
                acknowledgeMethod.invoke(manager, cmdId, response, timeout, message);
            } catch (ReflectiveOperationException x) {
                throw new SALException("Error acknowledging command", x);
            }
        }

        private int waitForCompletion(int cmdId, int timeout) throws SALException, TimeoutException {
            return commandHelper.waitForCompletion(cmdId, timeout);
        }

        private Duration waitForAck(int cmdId, int timeout) throws TimeoutException, SALException {
            return commandHelper.waitForAck(cmdId, timeout);   
        }
        
        @Override
        SALReceivedCommand pollForObject() throws SALException {
            try {
                S salInstance = getSALInstance();
                int cmdId = (int) acceptMethod.invoke(manager, salInstance);
                if (cmdId != 0) {
                    return new GenericReceivedCommand(cmdId, convertFromSAL(salInstance), this);
                } else {
                    return null;
                }
            } catch (ReflectiveOperationException ex) {
                throw new SALException("Error invoking accept method", ex);
            }
        }

        @Override
        SALCommandResponse send(SS message) throws SALException {
            try {
                int cmdId = (int) issueMethod.invoke(manager, convertToSAL(message));
                return new GenericCommandResponse(cmdId, this);
            } catch (ReflectiveOperationException x) {
                throw new SALException("Error converting command", x);
            }
        }
    }

    private static class SALObjectField<SS, S, T> {

        private final String name;
        private final Field salField;
        private final Method getter;
        private final Class getterType;
        private final boolean getterIsArray;
        private final boolean salFieldIsArray;
        private final Class salFieldType;

        private SALObjectField(String name, Field field, Method getter) {
            this.name = name;
            this.salField = field;
            this.getter = getter;
            this.getterIsArray = getter.getReturnType().isArray();
            this.getterType = getterIsArray ? getter.getReturnType().getComponentType() : getter.getReturnType();
            this.salFieldIsArray = salField.getType().isArray();
            this.salFieldType = salFieldIsArray ? salField.getType().getComponentType() : salField.getType();
        }

        /**
         * Copy this field between source and destination
         *
         * @param source The source (a simple-sal object)
         * @param destination The destination (a SAL object)
         * @throws ReflectiveOperationException
         */
        private void copy(SS source, S destination) throws ReflectiveOperationException {
            Object value;
            // Special handling for conversion of enums
            if (getterType.isEnum() && salFieldType.isPrimitive()) {
                if (getterIsArray) {
                    SALEnum[] array = (SALEnum[]) getter.invoke(source);
                    int arraySize = Array.getLength(array);
                    value = new int[arraySize];
                    for (int i = 0; i < arraySize; i++) {
                       Array.set(value, i, (int) array[i].getSALValue());
                    }
                } else {
                    value = (short) ((SALEnum) getter.invoke(source)).getSALValue();
                }
            } else {
                value = getter.invoke(source);
            }
            if (value.getClass().isArray() && Array.getLength(value) == 1 && !salFieldIsArray) {
                value = Array.get(value, 0);
            }
            salField.set(destination, value);
        }

        /**
         * Get the value of this field of the corresponding SAL object
         *
         * @param salInstance
         * @return The value
         * @throws ReflectiveOperationException
         */
        private Object get(S salInstance) throws ReflectiveOperationException {
            // Special handling for enums
            Object result;
            if (getterType.isEnum() && salFieldType.isPrimitive()) {
                if (salFieldIsArray) {
                    Object array = salField.get(salInstance);
                    int arraySize = Array.getLength(array);
                    result = Array.newInstance(getterType, arraySize);
                    for (int i = 0; i < arraySize; i++) {
                        Array.set(result, i, getEnum(Array.getInt(array, i)));
                    }
                } else {
                    result = getEnum(salField.getInt(salInstance));
                }
            } else {
                result = salField.get(salInstance);
            }
            if (getterIsArray && !salFieldIsArray) {
                Object array = Array.newInstance(getterType, 1);
                Array.set(array, 0, result);
                return array;
            }
            return result;
        }

        private Object getEnum(int value) throws IllegalArgumentException, ReflectiveOperationException, IllegalAccessException {
            for (Enum e : ((Class<Enum>) getterType).getEnumConstants()) {
                if (((SALEnum) e).getSALValue() == value) {
                    return e;
                }
            }
            throw new ReflectiveOperationException(String.format("Cannot find enumeration of class %s for value %d", getter.getReturnType(), value));
        }
    }

    /**
     * Base class representing a SALObject (commmand, event, telemetry) plus its
     * corresponding simple-sal equivalent. Also contains methods for converting
     * between the two representations.
     */
    private abstract class SALObject<SS, S> {

        private final SALType type;
        private final Class<SS> simpleSALClass;
        private final Class<S> salClass;
        private final List<SALObjectField> fields;
        private final Constructor simpleSALConstructor;
        private final S salInstance;

        SALObject(String[] line) throws ReflectiveOperationException, IOException {
            type = SALType.valueOf(line[0].toUpperCase());
            simpleSALClass = (Class<SS>) Class.forName(line[1]);
            salClass = (Class<S>) Class.forName(line[2]);
            List<Class> constructorArgTypes = new ArrayList<>();
            if (line.length == 3) {
                fields = Collections.EMPTY_LIST;
            } else {
                fields = new ArrayList<>();
                for (int i = 3; i < line.length; i += 2) {
                    final Class salType = parseFieldType(line[i]);
                    final Field field = salClass.getField(line[i + 1]);
                    // Allow for the fact the SAL does not distinguish between arrays of length 1 and scalars
                    if (field.getType() != salType && !(salType.isArray() && field.getType() == salType.getComponentType())) {
                        throw new IOException(String.format("Field %s does not have expected type %s", line[i + 1], line[i]));
                    }
                    final Method getter;
                    if (type == SALType.STATE && i == 5) {
                        getter = simpleSALClass.getMethod("getSubstate");
                        if (!getter.getReturnType().isEnum()) {
                            throw new IOException(String.format("Getter for %s does not have expected return type Enum", line[i + 1]));
                        }
                        constructorArgTypes.add(getter.getReturnType());
                    } else {
                        getter = simpleSALClass.getMethod(getGetterName(line[i + 1], salType));
                        // Allow for Enum <--> int/short conversion
                        if (getter.getReturnType().isEnum() && salType.isPrimitive()) {
                            // OK
                        } else if (getter.getReturnType().isArray() && getter.getReturnType().getComponentType().isEnum() && salType.isArray() && salType.getComponentType().isPrimitive()) {
                            // OK
                        } else if (getter.getReturnType() != salType) {
                            throw new IOException(String.format("Getter for %s does not have expected return type %s", line[i + 1], line[i]));
                        }
                        constructorArgTypes.add(getter.getReturnType());
                    }
                    fields.add(new SALObjectField(line[i + 1], field, getter));
                }
            }
            simpleSALConstructor = simpleSALClass.getConstructor(constructorArgTypes.toArray(new Class[0]));
            salInstance = salClass.newInstance();
        }

        private String getGetterName(String field, Class argType) {
            return (argType == Boolean.TYPE ? "is" : "get") + field.substring(0, 1).toUpperCase() + field.substring(1);
        }

        private Class parseFieldType(String typeString) throws ClassNotFoundException {

            boolean isArray = typeString.endsWith("[]");
            if (isArray) {
                typeString = typeString.substring(0, typeString.length() - 2);
            }
            switch (typeString) {
                case "boolean":
                    return isArray ? boolean[].class : Boolean.TYPE;
                case "int":
                    return isArray ? int[].class : Integer.TYPE;
                case "short":
                    return isArray ? short[].class : Short.TYPE;
                case "long":
                    return isArray ? long[].class : Long.TYPE;
                case "float":
                    return isArray ? float[].class : Float.TYPE;
                case "double":
                    return isArray ? double[].class : Double.TYPE;
                case "byte":
                    return isArray ? byte[].class : Byte.TYPE;
                case "String":
                    return String.class; // Note: SAL does not support string arrays
                default:
                    return Class.forName(typeString);
            }
        }

        SALType getType() {
            return type;
        }

        Class<SS> getSimpleSALClass() {
            return simpleSALClass;
        }

        Class<S> getSalClass() {
            return salClass;
        }

        List<SALObjectField> getFields() {
            return fields;
        }

        Constructor<SS> getSimpleSALConstructor() {
            return simpleSALConstructor;
        }

        S getSALInstance() {
            return salInstance;
        }

        S convertToSAL(SS source) throws ReflectiveOperationException {
            S result = getSalClass().newInstance();
            for (SALObjectField arg : this.getFields()) {
                arg.copy(source, result);
            }
            return result;
        }

        SS convertFromSAL(S source) throws ReflectiveOperationException {
            final List<SALObjectField> arguments = getFields();
            Object[] args = new Object[arguments.size()];
            int i = 0;
            for (SALObjectField arg : arguments) {
                args[i++] = arg.get(source);
            }
            return getSimpleSALConstructor().newInstance(args);
        }

        abstract Object pollForObject() throws SALException;

        abstract Object send(SS message) throws SALException;

    }

    private class GenericReceivedCommand extends SALReceivedCommand {

        private final SALCommandItem commandItem;

        public GenericReceivedCommand(int cmdId, SALCommand command, SALCommandItem commandItem) {
            super(cmdId, command);
            this.commandItem = commandItem;
        }

        @Override
        public void acknowledgeCommand(Duration timeToComplete) throws SALException {
            acknowledgeCommand(cmdInProgress, (int) timeToComplete.getSeconds(), "Ack : OK");
        }

        @Override
        public void reportComplete() throws SALException {
            acknowledgeCommand(cmdComplete, 0, "Done : OK");
        }

        @Override
        public void reportError(Exception ex) throws SALException {
            int errorCode = 0; 
            if (ex instanceof SALHasErrorCode) {
               errorCode = ((SALHasErrorCode) ex).getErrorCode();
            } 
            acknowledgeCommand(cmdFailed, errorCode, "Error : " + ex.getMessage());
        }

        @Override
        public void rejectCommand(String reason, int errorCode) throws SALException {
            acknowledgeCommand(cmdNoPerm, errorCode, "Ack : NO " + reason);
        }

        private void acknowledgeCommand(int response, int timeout, String message) throws SALException {
            commandItem.acknowledge(getCmdId(), response, timeout, message);
        }
    }

    private class GenericCommandResponse extends SALCommandResponse {

        private final int cmdId;
        private final SALCommandItem commandItem;
        private boolean commandComplete = false;

        public GenericCommandResponse(int cmdId, SALCommandItem commandItem) {
            this.cmdId = cmdId;
            this.commandItem = commandItem;
        }

        @Override
        public Duration waitForAck(Duration timeout) throws CommandFailedException, TimeoutException, SALException {
            Duration result =  commandItem.waitForAck(cmdId, (int) timeout.getSeconds());
            if (Duration.ZERO.equals(result)) commandComplete=true;
            return result;
        }

        @Override
        public int waitForCompletion(Duration timeout) throws CommandFailedException, SALException, TimeoutException {
            if (commandComplete) {
                return 303;
            }
            return commandItem.waitForCompletion(cmdId, (int) timeout.getSeconds());
        }
    }
}
