package org.lsst.sal.camera.main;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.sal.SALCommandResponse;
import org.lsst.sal.SALException;
import org.lsst.sal.SALReceivedCommand;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.SALCamera;
import org.lsst.sal.camera.command.ClearCommand;

/**
 * Demo main to allow Robs's testing framework to run
 * @author tonyj
 */
public class RobsMain {
    public static void main(String[] args) throws SALException, InterruptedException, ExecutionException, TimeoutException {
        if (args.length<1) {
            throw new IllegalArgumentException("Missing arguments");
        }
        ExecutorService executor = Executors.newFixedThreadPool(1);
        try (SALCamera camera = SALCamera.create()) {
            switch (args[0]) {
                case "sendCommand":
                    CameraCommand command = createCommand(args[1],remainingArgs(2,args));
                    SALCommandResponse response = camera.issueCommand(command);
                    response.waitForCompletion(Duration.ofSeconds(10));
                    System.out.println("Response:"+response);
                    break;
                case "receiveCommand":
                    Future<SALReceivedCommand> future = executor.submit(() -> camera.getNextCommand(Duration.ofSeconds(10)));
                    SALReceivedCommand result = future.get(10, TimeUnit.SECONDS);
                    result.reportComplete();
                    System.out.println("Result: "+result);
                    break;
                case "sendAndReceiveCommand":
                    // Wait for a command to be received
                    command = createCommand(args[1],remainingArgs(2,args));
                    future = executor.submit(() -> camera.getNextCommand(Duration.ofSeconds(10)));
                    response = camera.issueCommand(command);
                    result = future.get(10, TimeUnit.SECONDS);
                    result.reportComplete();
                    response.waitForCompletion(Duration.ofSeconds(1));
                    System.out.println("Result: "+result);
                    break;
                default:
                    throw new IllegalArgumentException("Unnrecognized action: "+args[0]);
            }
        } finally {
            executor.shutdown();
        }
    }

    private static CameraCommand createCommand(String command, String[] args) {
        switch (command) {
            case "clear":
                return new ClearCommand(Integer.parseInt(args[0]));
            default:
                throw new IllegalArgumentException("Unrecognized command: "+command);
        }
    }
    
    private static String[] remainingArgs(int offset, String[] args) {
        String[] result = new String[args.length-offset];
        System.arraycopy(args, offset, result, 0, args.length-offset);
        return result;
    }
    
}
