package org.lsst.sal;

import java.time.Duration;

/**
 * SALReceivedCommand is returned when accepting a command. It wraps the received 
 * command and provides methods for acknowledging receipt, and reporting completion
 * or error status.
 * @author tonyj
 * @param <C>
 */
public abstract class SALReceivedCommand<C extends SALCommand> {

    private final int cmdId;
    private final C command;

    protected SALReceivedCommand(int cmdId, C command) {
        this.cmdId = cmdId;
        this.command = command;
    }
    
    /**
     * (optionally) called by the receiver of the command to acknowledge the
     * command was received and accepted.
     *
     * @param timeToComplete The estimated time for completion of the command.
     * @throws org.lsst.sal.SALException If a communication exception occurs.
     */
    public abstract void acknowledgeCommand(Duration timeToComplete) throws SALException;

    /**
     * Called by the receiver of the command to indicate the command has been
     * completed successfully.
     * @throws org.lsst.sal.SALException If a communication exception occurs.
     */
    public abstract void reportComplete() throws SALException;

    /**
     * Called by the receiver to indicate that the command execution failed.
     *
     * @param ex The exception containing the reason for failure.
     * @throws org.lsst.sal.SALException If a communication exception occurs.
     */
    public abstract void reportError(Exception ex) throws SALException;

    /**
     * Called by the receiver of the command to indicate that the command was
     * rejected.
     *
     * @param reason The reason for rejection.
     * @throws org.lsst.sal.SALException If a communication exception occurs.
     */
    public abstract void rejectCommand(String reason) throws SALException;

    /**
     * Get the command id of the received command
     * @return The command id, a positive integer,
     */
    public int getCmdId() {
        return cmdId;
    }
    
    /**
     * Get the actual command received,
     * @return The command
     */
    public C getCommand() {
        return command;
    }
}
