package org.lsst.sal;

import java.time.Duration;

/**
 * A generic interface to SAL.
 * 
 * @author tonyj
 * @param <C> The base class for all commands sent and received by this implementation
 * @param <E> The base class for all events sent and received by this implementation
 * @param <T> The base class for all telemetry sent and received by this implementation
 */
public interface SAL<C extends SALCommand, E extends SALEvent, T extends SALTelemetry> extends AutoCloseable {

    /**
     * Fetch the next command, or timeout if specified duration is exceeded.
     * After receiving a command the receiver is responsible for calling methods
     * in returned SALReceivedCommand to acknowledge and respond to the command.
     *
     * @param timeout The maximum time to wait before timing out
     * @return The next command received, or <code>null</code> if the wait
     * times-out.
     * @throws org.lsst.sal.SALException If a communication or similar
     * error occurs
     */
    SALReceivedCommand<C> getNextCommand(Duration timeout) throws SALException;

    /**
     * Issue a command,
     *
     * @param command The command to issue.
     * @return The CommandResponse, which can be used to obtain the response(s)
     * from the client.
     * @throws SALException If a communication or similar error occurs
     */
    SALCommandResponse issueCommand(C command) throws SALException;

    /**
     * Sends an event
     *
     * @param event The event to send
     * @throws SALException If a communication or similar error occurs
     */
    void logEvent(E event) throws SALException;

    E getNextEvent(Duration timeout) throws SALException;

    void sendTelemetry(T telemetry) throws SALException;

    T getTelemetry(Duration timeout) throws SALException;

    @Override
    public void close() throws SALException;
}
