package org.lsst.ccs.web.sequencer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents the time history of a waveform generated by the sequencer. Since
 * this may generate many points, code is also provided for converting the full
 * history into a sparsified history, in which the average value of each output
 * line is stored.
 * @author tonyj
 */
class WaveformHistory {
    
    private static final int MAX_HISTORY_SIZE = 10_000_000;
    private final List<WaveformHistoryEntry> history = new ArrayList<>();

    void add(WaveformHistoryEntry entry) {
        if (history.size() > MAX_HISTORY_SIZE) {
            throw new RuntimeException("Max History size exceeded");
        }
        history.add(entry);
    }

    private int computeBinningFactor(int totalBins) {
        return totalBins > 10000000 ? 10000 : totalBins > 1000000 ? 1000 : totalBins > 100000 ? 100 : totalBins > 10000 ? 10 : totalBins > 2000 ? 2 : 1;
    }

    Map<String, Object> possiblySparsify(Map<String, Integer> labels) {
        Map<String, Object> result = new HashMap();
        int binning = computeBinningFactor(history.size());
        if (binning > 1) {
            List<TransmogrifiedHistoryEntry> sparsifiedData = sparsifyData(history, labels, binning);
            result.put("binning", binning);
            result.put("sparsified", sparsifiedData);
        } else {
            result.put("data", history);
        }
        return result;
    }

    List<TransmogrifiedHistoryEntry> zoom(Map<String, Integer> labels, int min, int max, int binFactor) {
        List<WaveformHistoryEntry> subList = history.subList(min, Math.min(max, history.size()));
        return sparsifyData(subList, labels, binFactor);
    }

    private List<TransmogrifiedHistoryEntry> sparsifyData(List<WaveformHistoryEntry> data, Map<String, Integer> labels, int binFactor) {
        // Currently we are sparsifying based on # bins, not binWidth
        List<TransmogrifiedHistoryEntry> sparsifiedData = new ArrayList<>();
        // Cache bitOrder to speed up bit reordering
        int[] bitOrder = new int[labels.size()];
        int l = 0;
        for (int bit : labels.values()) {
            bitOrder[l++] = bit;
        }
        long binStart = 0;
        int bins = 0;
        int[] sum = new int[32];
        for (int i = 0; i < data.size(); i++) {
            WaveformHistoryEntry h = data.get(i);
            if (bins == 0) {
                binStart = h.getTime();
            }
            int out = h.getOut();
            for (int bit = 0; bit < 32 && out != 0; bit++) {
                sum[bit] += (out & 1);
                out >>= 1;
            }
            bins++;
            // Note, this ensures we keep the last point unmodified
            if (bins == binFactor || i >= data.size() - 2) {
                float[] averages = new float[bitOrder.length];
                for (int k = 0; k < bitOrder.length; k++) {
                    int bit = bitOrder[k];
                    averages[k] = k * 2 + ((float) sum[bit]) / bins;
                }
                TransmogrifiedHistoryEntry tw = new TransmogrifiedHistoryEntry(binStart, averages);
                sparsifiedData.add(tw);
                Arrays.fill(sum, 0);
                bins = 0;
            }
        }
        return sparsifiedData;
    }

    static class WaveformHistoryEntry {

        private final int currentFunction;
        private final long time;
        private final int out;

        public WaveformHistoryEntry(int currentFunction, long time, int out) {
            this.currentFunction = currentFunction;
            this.time = time;
            this.out = out;
        }

        public int getCurrentFunction() {
            return currentFunction;
        }

        public long getTime() {
            return time;
        }

        public int getOut() {
            return out;
        }

    }

    static class TransmogrifiedHistoryEntry {

        private final long binStart;
        private final float[] averages;

        private TransmogrifiedHistoryEntry(long binStart, float[] averages) {
            this.binStart = binStart;
            this.averages = averages;
        }

        public long getBinStart() {
            return binStart;
        }

        public float[] getAverages() {
            return averages;
        }
    }
}
