package org.lsst.ccs.web.sequencer;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import javax.ws.rs.core.MultivaluedMap;
import org.lsst.ccs.drivers.reb.Sequencer;
import org.lsst.ccs.drivers.reb.sim.AddressSpace;
import org.lsst.ccs.drivers.reb.sim.SequencerSimulation;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 * Encapsulates the behavior of the sequencer simulator.
 * @author tonyj
 */
class SequencerEncapsulation {
    
    private AddressSpace addressSpace;
    private SequencerSimulation sequencer;
    private FPGA2Model model;
    private boolean sequencerReady = false;
    private int max_repetitions = 3;
    private WaveformHistory history;

    private Map<String, Object> runSequencer() throws InterruptedException, ExecutionException {
        WaveformHistoryListener listener = new WaveformHistoryListener();
        sequencer.addWaveformListener(listener);
        sequencer.addFunctionListener(listener);
        sequencer.setMaxFunctionRepetitions(max_repetitions);
        sequencer.setMaxSubroutineRepetitions(max_repetitions);
        sequencer.setRenormalization(max_repetitions);
        addressSpace.write(Sequencer.REG_SEQ_ERROR_RESET, 1);
        try {
            Future<SequencerSimulation.FinalState> future = sequencer.start();
            SequencerSimulation.FinalState state = future.get();
            System.out.println("State=" + state.getStateAtEnd());
        } finally {
            sequencer.removeWaveformListener(listener);
            sequencer.removeFunctionListener(listener);
        }
        history = listener.getHistory();
        return history.possiblySparsify(model.getChannels());
    }

    Map<String, Object> read(InputStream in) throws IOException {
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        addressSpace = new AddressSpace();
        sequencer = new SequencerSimulation(addressSpace);
        model = builder.compileFile(in);
        model.getMemoryMap().stream().forEach((data) -> {
            addressSpace.write(data.getAddress(), data.getValue());
        });
        sequencerReady = false;
        Map<String, Object> response = new HashMap<>();
        response.put("constants", model.getMetadata());
        response.put("channels", model.getChannels());
        response.put("functions", model.getFunctionAddresses());
        response.put("subroutines", model.getSubroutineAddresses());
        response.put("mains", model.getMainAddresses());
        response.put("pointers", model.getPointers());
        return response;
    }

    Map<String, Object> run(String key, int address) throws InterruptedException, ExecutionException {
        switch (key) {
            case "function":
                int function = address;
                int program = Sequencer.REG_SEQ_PROGRAM + Sequencer.SEQ_MAX_PROGRAM - 2;
                addressSpace.write(program++, Sequencer.SEQ_OPC_EXECUTE << Sequencer.SEQ_PRG_V_OPCODE | (function << 24) | 1);
                addressSpace.write(program++, Sequencer.SEQ_OPC_END_PROG << Sequencer.SEQ_PRG_V_OPCODE);
                addressSpace.write(Sequencer.REG_SEQ_START_ADDR, Sequencer.SEQ_MAX_PROGRAM - 2);
                break;
            case "subroutine":
                int subroutine = address;
                program = Sequencer.REG_SEQ_PROGRAM + Sequencer.SEQ_MAX_PROGRAM - 2;
                addressSpace.write(program++, Sequencer.SEQ_OPC_JUMP << Sequencer.SEQ_PRG_V_OPCODE | (subroutine << 16) | 1);
                addressSpace.write(program++, Sequencer.SEQ_OPC_END_PROG << Sequencer.SEQ_PRG_V_OPCODE);
                addressSpace.write(Sequencer.REG_SEQ_START_ADDR, Sequencer.SEQ_MAX_PROGRAM - 2);
                break;
            case "main":
                addressSpace.write(Sequencer.REG_SEQ_START_ADDR, address);
                break;
            default:
                throw new IllegalArgumentException("Unrecognized key " + key);
        }
        sequencerReady = true;
        return runSequencer();
    }

    Map<String, Object> set(MultivaluedMap<String, String> parameters) throws InterruptedException, ExecutionException {
        String message = "?";
        String max = parameters.getFirst("max-repetitions");
        if (max != null) {
            max_repetitions = Integer.parseInt(max);
            message = "OK";
        }
        if (model != null) {
            for (FPGA2Model.PointerInfo pointer : model.getPointers()) {
                String value = parameters.getFirst(pointer.getName());
                if (value != null) {
                    addressSpace.write(pointer.getAddress(), Integer.parseInt(value));
                    message = "OK";
                }
            }
        }
        Map<String, Object> result = new HashMap<>();
        result.put("message", message);
        if (sequencerReady) {
            result.putAll(runSequencer());
        }
        return result;
    }

    Object zoom(int min, int max, int bins) {
        return history.zoom(model.getChannels(), min, max, bins);
    }
    
    /**
     * Listens on the sequencer and builds a WaveformHistory
     */
    private static class WaveformHistoryListener implements SequencerSimulation.WaveformListener, SequencerSimulation.FunctionListener {

        private int currentFunction = 0;
        private long accumulatedNanos = 0;
        private final WaveformHistory history = new WaveformHistory();

        @Override
        public void transition(int oldState, int newState, int nanos) {

            history.add(new WaveformHistory.WaveformHistoryEntry(currentFunction, accumulatedNanos, newState));
            accumulatedNanos += nanos;
        }

        @Override
        public void functionCalled(int function, boolean infiniteLoop, int repetitions) {
            currentFunction = function;
        }

        public WaveformHistory getHistory() {
            return history;
        }
    }
}
