package org.lsst.ccs.utilities.conv;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.time.format.DateTimeParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class is responsible for converting strings to objects of a particular
 * class. It is used to convert command arguments into objects to be passed to
 * the corresponding method. The current implementation is very simple, dealing
 * only with built in types, or types with constructors which take a string. It
 * would be possible to add back in the cliche functionality which allowed extra
 * input converters to be registered with the input conversion engine.
 *
 * Note that in a distributed system argument conversion is only done on the
 * remote (server) end, so any special classes used for arguments only need to
 * be present on the remote (server). In CCS terminology this means that the
 * classes only need to be present in the subsystem.
 *
 * @author tonyj
 */
public class InputConversionEngine {

    /**
     * Convert the input string to an object of the given class.
     * The second argument is a java.lang.reflect.Type. The reason is so that
     * we can use Java Generics when we are dealing with templated classes.
     * @param arg The string to be converted
     * @param inputType The Type that the string should be converted to
     * @return The converted argument
     * @throws TypeConversionException If the string cannot be converted to the requested type
     *
     */
    static public Object convertArgToType(String arg, Type inputType) throws TypeConversionException {
        if (TypeUtils.NULL_STR.equals(arg)) return null;
        Class type = inputType instanceof Class ? (Class) inputType : null;
        boolean isParameterized = inputType instanceof ParameterizedType;
        if (isParameterized) {
            type = (Class) ((ParameterizedType) inputType).getRawType();
        }
        if (type == null) {
            throw new RuntimeException("Error: Could not find type for " + inputType);
        }
        try {
            if (type.equals(String.class) || type.isInstance(arg)) {
                return arg;
            } else if (type.equals(Integer.class) || type.equals(Integer.TYPE)) {
                // Want to handle 0x800000000, e.g., which is an invalid int
                long value = Long.decode(arg);
                long sign = value >> 32;
                if (sign != 0 && sign != -1) {
                    throw new NumberFormatException();
                }
                return (int) value;
            } else if (type.equals(Long.class) || type.equals(Long.TYPE)) {
                return Long.decode(arg);
            } else if (type.equals(Short.class) || type.equals(Short.TYPE)) {
                return Short.valueOf(arg);
            } else if (type.equals(Byte.class) || type.equals(Byte.TYPE)) {
                return Byte.valueOf(arg);
            } else if (type.equals(Double.class) || type.equals(Double.TYPE)) {
                return Double.valueOf(arg);
            } else if (type.equals(Float.class) || type.equals(Float.TYPE)) {
                return Float.valueOf(arg);
            } else if (type.equals(Boolean.class) || type.equals(Boolean.TYPE)) {
                // Boolean.parseBoolean treats anything it does not recognize as true, which does not
                // seem appropriate here, so instead
                if ("true".equalsIgnoreCase(arg)) {
                    return Boolean.TRUE;
                }
                if ("false".equalsIgnoreCase(arg)) {
                    return Boolean.FALSE;
                }
                throw new TypeConversionException("Error: Can't convert string '%s' to Boolean", arg);
            } else if (type.isEnum()) {
                //FIXME: This is case sensitive, is this what we want?
                //FIXME: Would be nice to generate exception with list of legal values?
                try {
                    return Enum.valueOf(type, arg.toUpperCase());
                } catch (IllegalArgumentException x) {
                    return Enum.valueOf(type, arg);
                }
            } else if (type.isArray()) {
                //Array Conversion happens here
                List<String> tokens = splitOnBrackets(arg);
                int n = tokens.size();
                Class arrayType = type.getComponentType();
                Object array = Array.newInstance(arrayType, n);
                int count = 0;
                for (String token : tokens) {
                    Array.set(array, count++, convertArgToType(token, arrayType));
                }
                return array;
            } else if (java.util.List.class.isAssignableFrom(type)) {
                //List Conversion happens here.
                Type listType = isParameterized ? ((ParameterizedType) inputType).getActualTypeArguments()[0] : Object.class;
                ArrayList l = new ArrayList();
                List<String> tokens = splitOnBrackets(arg);
                for (String token : tokens) {
                    l.add(convertArgToType(token, listType));
                }
                return l;
            } else if (java.util.Set.class.isAssignableFrom(type)) {
                //Set Conversion happens here.
                Type listType = isParameterized ? ((ParameterizedType) inputType).getActualTypeArguments()[0] : Object.class;
                Set s = new HashSet();
                List<String> tokens = splitOnBrackets(arg);
                for (String token : tokens) {
                    s.add(convertArgToType(token, listType));
                }
                return s;
            } else if (java.util.Map.class.isAssignableFrom(type)) {
                //Map Conversion happens here.
                Type keyType = isParameterized ? ((ParameterizedType) inputType).getActualTypeArguments()[0] : Object.class;
                Type valueType = isParameterized ? ((ParameterizedType) inputType).getActualTypeArguments()[1] : Object.class;
                Map map = new HashMap();
                List<String> tokens = splitOnBrackets(arg);
                for (String token : tokens) {
                    String keyVal = token;
                    int split = keyVal.indexOf(":");
                    if (split < 0) {
                        if ( keyVal.isEmpty() ) {
                            continue;
                        }
                        throw new IllegalArgumentException("Error: cannot parse key-value pair " + keyVal + ". Key and Value must be separated by a column");
                    }
                    map.put(convertArgToType(keyVal.substring(0, split), keyType), convertArgToType(keyVal.substring(split + 1), valueType));
                }
                return map;
            } else if (type.equals(Instant.class)) {
                return Instant.parse(arg);
            } else if (type.equals(Duration.class)) {
                return Duration.parse(arg);
            } else {
                Constructor c = type.getConstructor(String.class);
                return c.newInstance(arg);

            }
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException ex) {
            throw new TypeConversionException("Error: Can't instantiate class %s using string '%s'", ex, type.getName(), arg);
        } catch (IllegalArgumentException | NoSuchMethodException | DateTimeParseException e) {
            throw new TypeConversionException("Error: Can't convert string '%s' to class %s", e, arg, type.getName());
        }
    }

    /**
     * Utility function to split string on square brackets and commas as described in
     * https://jira.slac.stanford.edu/browse/LSSTCCS-29
     */
    static List<String> splitOnBrackets(String input) {
        input = input.trim();
        List<String> list = new ArrayList<>();
        if ( input.equals("[]") ) {
            return list;
        }
        StringBuilder token = new StringBuilder("");
        int nBrackets = 0;
        for (int i = 0; i < input.length(); i++) {
            char c = input.charAt(i);
            if (c == '[') {
                nBrackets++;
            } else if (c == ']') {
                nBrackets--;
            }

            boolean newToken = nBrackets == 0;
            if (nBrackets == 1) {
                if (c == '[') {
                    continue;
                } else if (c == ',') {
                    newToken = true;
                }
            }

            if (newToken) {
                list.add(token.toString().trim());
                token.setLength(0);
            } else {
                token.append(c);
            }

        }

        return list;
    }

}
