package org.lsst.ccs.utilities.scheduler;

import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Thread factory to be used by executors that need to customize thread name, group, uncaught exception handling, and daemon status.
 *
 * @author onoprien
 */
public class BasicThreadFactory implements ThreadFactory {

// -- Fields : -----------------------------------------------------------------
    
        private static final AtomicInteger FACTORY_NUMBER = new AtomicInteger(1);
        
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;
        private final boolean daemon;
        private final Thread.UncaughtExceptionHandler еxceptionHandler;

        
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates factory that makes non-daemon threads in the current thread group with default exception handling.
     * 
     * @param name Thread name prefix: threads will be named "name-N".
     *             If {@code null}, threads will be named BasicThreadFactory-K-N".
     */
    public BasicThreadFactory(String name) {
        this(name, null, false, null);
    }
    
    /**
     * Creates factory that makes threads in the current thread group with default exception handling.
     * 
     * @param name Thread name prefix: threads will be named "name-N".
     *             If {@code null}, threads will be named BasicThreadFactory-K-N".
     * @param daemon Whether or not threads made by this factory are daemons.
     */
    public BasicThreadFactory(String name, boolean daemon) {
        this(name, null, daemon, null);
    }
    
    /**
     * Creates thread factory with default exception handling.
     * 
     * @param name Thread name prefix: threads will be named "name-N".
     *             If {@code null}, threads will be named BasicThreadFactory-K-N".
     * @param group Thread group. If {@code null}, the group of the thread that creates
     *              this factory will be used, unless {@code SecurityManager} exists.
     * @param daemon Whether or not threads made by this factory are daemons.
     */
    public BasicThreadFactory(String name, ThreadGroup group, boolean daemon) {
        this(name, group, daemon, null);
    }
    
    /**
     * Creates thread factory.
     * 
     * @param name Thread name prefix: threads will be named "name-N".
     *             If {@code null}, threads will be named BasicThreadFactory-K-N".
     * @param group Thread group. If {@code null}, the group of the thread that creates
     *              this factory will be used, unless {@code SecurityManager} exists.
     * @param daemon Whether or not threads made by this factory are daemons.
     * @param еxceptionHandler Uncaught exception handler, or {@code null} if default handling should be used.
     */
    public BasicThreadFactory(String name, ThreadGroup group, boolean daemon, Thread.UncaughtExceptionHandler еxceptionHandler) {
        if (group == null) {
            SecurityManager s = System.getSecurityManager();
            this.group = (s != null) ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
        } else {
            this.group = group;
        }
        namePrefix = name == null ? "BasicThreadFactory-" + FACTORY_NUMBER.getAndIncrement() + "-" : name + "-";
        this.daemon = daemon;
        this.еxceptionHandler = еxceptionHandler;
    }

    
// -- Making threads : ---------------------------------------------------------
    
    @Override
    public Thread newThread(Runnable r) {
        Thread t = new Thread(group, r, namePrefix + threadNumber.getAndIncrement(), 0);
        try {
            if (t.isDaemon() != daemon) {
                t.setDaemon(daemon);
            }
            if (t.getPriority() != Thread.NORM_PRIORITY) {
                t.setPriority(Thread.NORM_PRIORITY);
            }
            if (еxceptionHandler != null) {
                t.setUncaughtExceptionHandler(еxceptionHandler);
            }
        } catch (SecurityException x) {
        }
        return t;
    }
    
}
