package org.lsst.ccs.utilities.pattern;

import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class to define patterns for file names and dynamically resolve and
 * inject quantities into it.
 *
 * A pattern variable is defined by the following sequence: ${VARIABLE}.
 * If VARIABLE has been added as a key with a corresponding value it will be
 * dynamically substituted at runtime in the file name. If VARIABLE is not set on this
 * class, then its pattern will be left untouched.
 *
 * @author turri
 */
public class FileNamePatternUtils {

    private final static Pattern variablePattern = Pattern.compile("(\\{([\\w/]+){1}}).*");    
    
    /**
     * Resolve a fileNamePatter by replacing escape sequences of the form ${VARIABLE}
     * with the stored values.
     *
     * @param fileNamePattern The fileNamePattern to resolve.
     * @param map A Map object containing the keys and values to resolve
     *            the variable components in the provided pattern
     * @return The resolved fileName.
     */
    public static String resolveFileName(String fileNamePattern, Map<String,String> map) {
        Properties p = new Properties();
        for ( String key : map.keySet() ) {
            p.setProperty(key, map.get(key));            
        }
        return resolveFileName(fileNamePattern, p);
    }
    
    /**
     * Resolve a fileNamePatter by replacing escape sequences of the form ${VARIABLE}
     * with the stored values.
     *
     * @param fileNamePattern The fileNamePattern to resolve.
     * @param props A Properties object containing the keys and values to resolve
     *              the variable components in the provided pattern
     * @return The resolved fileName.
     */
    public static String resolveFileName(String fileNamePattern, Properties props) {
        String[] parts = fileNamePattern.split("\\$");
        if (parts.length < 2) {
            return fileNamePattern;
        }
        String resolvedFileName = parts[0];
        for (int i = 1; i < parts.length; i++) {
            resolvedFileName += resolvePart(parts[i], props);
        }        
        return resolvedFileName;
    }

    private static String resolvePart(String part, Properties props) {
        Matcher m = variablePattern.matcher(part);
        if (m.matches()) {
            String group = m.group(1);
            String variableName = group.substring(1, group.length() - 1);
            String value = props.getProperty(variableName, null);
            if ( value != null ) {
                return part.replace("{"+variableName+"}", value);
            }
        }         
        return "$"+part;
    }

}
