package org.lsst.ccs.utilities.logging;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Date;
import java.util.logging.ErrorManager;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

/**
 * A Log File Handler that rolls every day.
 * 
 * @author The LSST CCS team
 */
public class DailyRollingFileHandler extends Handler {
    
    private String pattern = "/tmp/output.log";
    private int count = 1;
    private FileHandler innerFileHandler;
    private long nextRollover = -1;

    public DailyRollingFileHandler() {           
        String className = this.getClass().getCanonicalName();
        pattern = LogPropertiesLoader.loaderGetStringProperty(className+".pattern", pattern);
        count = LogPropertiesLoader.loaderGetIntProperty(className+".count", 1);
        
        Formatter formatter = LogPropertiesLoader.loaderGetFormatterProperty(className + ".formatter", new TextFormatter());
        this.setFormatter(formatter);
        Level level = LogPropertiesLoader.loaderGetLevelProperty(className + ".level", Level.WARNING);
        this.setLevel(level);
    }
    
    private synchronized void createFileHandlerForTodayIfNeeded(long millis) {
        if (millis < nextRollover) {
            return;
        }
        Instant instant = Instant.ofEpochMilli(millis);
        ZoneId zoneId = ZoneId.systemDefault();
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        ZonedDateTime zdtStart = zdt.toLocalDate().atStartOfDay(zoneId);
        ZonedDateTime zdtTomorrowStart = zdtStart.plusDays(1);
        nextRollover = zdtTomorrowStart.toInstant().toEpochMilli();
        try {
            if (innerFileHandler != null) {
                innerFileHandler.close();
            }
            innerFileHandler = new FileHandler(fileName(millis), 0, count, count==1);
            innerFileHandler.setLevel(getLevel());
            innerFileHandler.setEncoding(getEncoding());
            innerFileHandler.setFilter(getFilter());
            if ( getFormatter() != null ) {
                innerFileHandler.setFormatter(getFormatter());
            }
            innerFileHandler.setErrorManager(getErrorManager());
        } catch (IOException e) {
            this.reportError(null, e, ErrorManager.WRITE_FAILURE);
        }
    }
    
    
    @Override
    public synchronized void publish(LogRecord r) {
        if (isLoggable(r)) {
            createFileHandlerForTodayIfNeeded(r.getMillis());
            innerFileHandler.publish(r);
        }
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() {
        super.setLevel(Level.OFF);
        if ( innerFileHandler != null ) {
            innerFileHandler.close();
        }
    }

    String fileName(long millis) {
        String date = new SimpleDateFormat("yyyyMMdd").format(new Date(millis));
        return pattern.replace(".log","-"+date+".log");
    }
    
    long getNextRollover() {
        return nextRollover;
    }
}
