package org.lsst.ccs.utilities.reflect;

import java.lang.reflect.Constructor;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 *
 * @author The LSST CCS Team
 */
public class ConstructorUtils {
    
    /**
     * This variable is to be used when it is not possible to determine the
     * class of the argument because its value is null.
     * 
     */
    public static Class NULL_CLASS = NullObject.class;
    
    private ConstructorUtils() {
    }
    
    /**
     * Find a constructor that can be used given an array of argument types.
     * It extends the capabilities of the isAssignableFrom method which does not
     * cover primitive types in a context of passing arguments to the constructor.
     * It first looks for non varargs constructor, and then for varargs ones.
     * @param klass the class to introspect
     * @param argumentsClasses the arguments
     * @return a constructor that matches the given argument types
     */
    public static Constructor getMatchingConstructor(Class<?> klass, Class<?>[] argumentsClasses ) {
        Constructor[] ctors = klass.getConstructors();
        Set<Constructor> varargsCtors = new HashSet<>();

        List<Constructor> validConstructors = new ArrayList<>();
        // Go over the constructors :
        for (Constructor c : ctors) {
            if (c.isVarArgs()) {
                varargsCtors.add(c);
            } else {
                if (constructorMatches(c, argumentsClasses)) {
                    validConstructors.add(c);
                }
            }
        }
        if (validConstructors.size() == 1) return validConstructors.get(0);
        if (validConstructors.size() > 1 )
            throw new AmbiguousConstructorException("ambiguous constructors : "
                    + validConstructors);
        // Go over the varargs constructors
        for (Constructor c : varargsCtors) {
            if (varArgsConstructorMatches(c, argumentsClasses)) {
                validConstructors.add(c);
            }
        }
        if (validConstructors.size() == 1) return validConstructors.get(0);
        if (validConstructors.size() > 1 )
            throw new AmbiguousConstructorException("ambiguous constructors : "
                    + validConstructors);
        return null;
    }
    
    
    /**
     * In the context of a constructor : It checks if an object of the class 
     * from can be passed as an argument which is of type to.
     * @param to the type of the argument in the constructor
     * @param from the type of the value that is to be assigned as argument of
     * the constructor.
     * @return 
     */
    private static boolean typeMatch(Class to, Class from) {
        if (to.isAssignableFrom(from)) return true;
        if ( ! to.isPrimitive() && from == NULL_CLASS ) return true;        
        if (to.isPrimitive() && from.isPrimitive()) {
            return bothDifferentPrimitiveTypeMatch(to, from);
        }
        if (to.isPrimitive()) {
            return assignToPrimitiveTypeMatch(to, from);
        } else if (from.isPrimitive()) {
            return assignFromPrimitiveTypeMatch(to, from);
        }
        return false;
    }
    
    /**
     * To is non primitive and from is primitive.
     * REVIEW : does it return true if and only if to==Object.class ? Are there 
     * other cases ?
     * @param to a non primitive type.
     * @param from a primitive type
     * @return true if to is assignable from the wrapper class from. 
     */
    private static boolean assignFromPrimitiveTypeMatch(Class to, Class from) {
        return (int.class.equals(from) && (to.isAssignableFrom(Integer.class))
                || (long.class.equals(from) && to.isAssignableFrom(Long.class))
                || (double.class.equals(from) && to.isAssignableFrom(Double.class))
                || (char.class.equals(from) && to.isAssignableFrom(Character.class))
                || (short.class.equals(from) && to.isAssignableFrom(Short.class))
                || (boolean.class.equals(from) && to.isAssignableFrom(Boolean.class))
                || (byte.class.equals(from) && to.isAssignableFrom(Byte.class))
                );
        
    }
    
    /**
     * 'To' is a primitive type and 'from' is not.
     * @param to a primitive type
     * @param from a non primitive type
     * @return true if from is the wrapper class of a primitive type and its 
     * corresponding primitive type can be promoted to the type 'to'.
     */
    private static boolean assignToPrimitiveTypeMatch(Class to, Class from) {
        return (
                (boolean.class.equals(to)   &&  from.equals(Boolean.class))
                || (byte.class.equals(to)   && (from.equals(Byte.class)))
                || (char.class.equals(to)   && (from.equals(Character.class)))
                || (short.class.equals(to)  && (from.equals(Byte.class) || from.equals(Short.class)))
                || (int.class.equals(to)    && (from.equals(Byte.class) || from.equals(Short.class) || from.equals(Character.class) || from.equals(Integer.class)  ))
                || (long.class.equals(to)   && (from.equals(Byte.class) || from.equals(Short.class) || from.equals(Character.class) || from.equals(Integer.class) || from.equals(Long.class)))
                || (double.class.equals(to) && (from.equals(Byte.class) || from.equals(Short.class) || from.equals(Character.class) || from.equals(Integer.class) || from.equals(Long.class) || from.equals(Double.class)))
                );
    }
    
    /**
     * isAssignableFrom returns true for primitive type iff the class objects are
     * exactly the same. In practice calling a constructor can operate type conversions.
     * @param to a primitive type
     * @param from a primitive type
     * @return true if 'from' can be promoted to the type 'to'.
     */
    private static boolean bothDifferentPrimitiveTypeMatch(Class to, Class from) {
        if (boolean.class.equals(from)) return false;
        if (double.class.equals(to)) return true;
        return (long.class.equals(to) && !double.class.equals(from)
                || int.class.equals(to) && !(double.class.equals(from)||long.class.equals(from))
                || short.class.equals(to) && (char.class.equals(from) || byte.class.equals(from))
                );
    }
    
    private static boolean constructorMatches(Constructor c, Class[] parameterClasses) {
        Class<?>[] constructorArgumentClasses = c.getParameterTypes();
        if (!(constructorArgumentClasses.length == parameterClasses.length)) return false;
        for (int i=0; i<parameterClasses.length; i++) {
            if (!typeMatch(constructorArgumentClasses[i], parameterClasses[i]))
                return false;
        }
        return true;
    }
    
    private static boolean varArgsConstructorMatches(Constructor c, Class[] parameterClasses) {
        Parameter[] arguments = c.getParameters();
        // loop over the constructor arguments except the varargs one
        for (int i=0; i<arguments.length-1; i++) {
            if (!typeMatch(arguments[i].getType(), parameterClasses[i])) return false;
        }
        int varargsIndex = arguments.length -1;
        Class varargType = arguments[varargsIndex].getType().getComponentType();
        for (int i = varargsIndex ; i<parameterClasses.length; i++) {
            if (!typeMatch(varargType, parameterClasses[i])) return false;
        }
        return true;
    }
    
    /**
     * Returns the type of the parameter at a given index of the given constructor
     * @param c the constructor
     * @param index the index of the parameter
     * @return the resolved type.
     */
    public static Type getParameterType(Constructor c, int index) {
        int parmLength = c.getParameterCount();
        if (index < parmLength - 1) {
            return c.getParameters()[index].getParameterizedType();
        }
        else {
            java.lang.reflect.Parameter lastParm = c.getParameters()[parmLength-1];
            if (lastParm.isVarArgs()) {
                // Cannot access parameterized type of a varargs ?
                return lastParm.getType().getComponentType();
            } else {
                if (index == parmLength - 1) {
                    return lastParm.getParameterizedType();
                }
                throw new IllegalArgumentException("index is greater than the number of arguments");
            }
        }
    }

    private static class NullObject {
        
    }    
}
