package org.lsst.ccs.utilities.dsp;

/**
 * 
 * @author aubourg
 * 
 */
public class FilterFactory {

	public enum FilterType {
		LOW_PASS, BAND_PASS, HIGH_PASS
	};

	public static FIRFilter getHighPassFIR(double rate, double attenuation,
			double f1) {
		return getFIRFilter(rate, attenuation, f1, -1, FilterType.HIGH_PASS);
	}

	public static FIRFilter getBandPassFIR(double rate, double attenuation,
			double f1, double f2) {
		return getFIRFilter(rate, attenuation, f1, f2, FilterType.BAND_PASS);
	}

	public static FIRFilter getLowPassFIR(double rate, double attenuation,
			double f2) {
		return getFIRFilter(rate, attenuation, -1, f2, FilterType.LOW_PASS);
	}

	public static FIRFilter getFIRFilter(double rate, double attenuation,
			double f1, double f2, FilterType type) {
		FilterSpecs spec = new FilterSpecs();
		spec.setFN(rate / 2).setAttenuation(attenuation).setLowFrequency(f1)
				.setHighFrequency(f2).setType(type);
		if (f1 > 0) {
			spec.setTransitionBandwidth(f1 / 5);
		} else {
			spec.setTransitionBandwidth(f2 / 5);
		}
		return getFIRFilter(spec);
	}

	public static FIRFilter getFIRFilter(FilterSpecs spec) {
		if (spec.order < 1) {
			spec.order = estimatedFIROrder(spec);
		}
		// estimate Kaiser window parameter (alpha):
		double alpha = 0;
		if (spec.getAttenuation() >= 50.0) {
			alpha = 0.1102 * (spec.getAttenuation() - 8.7);
		} else if (spec.getAttenuation() > 21.0) {
			alpha = 0.5842
					* Math.exp(0.4 * Math.log(spec.getAttenuation() - 21.0))
					+ 0.07886 * (spec.getAttenuation() - 21.0);
		}
		if (spec.getAttenuation() <= 21.0) {
			alpha = 0.0;
		}
		// window function values
		double I0alpha = I0(alpha);
		int m = spec.order / 2;
		double[] win = new double[m + 1];
		for (int n = 1; n <= m; n++) {
			win[n] = I0(alpha * Math.sqrt(1.0 - sqr((double) n / m))) / I0alpha;
		}
		double w0 = 0.0;
		double w1 = 0.0;
		switch (spec.type) {
		case LOW_PASS:
			w0 = 0.0;
			w1 = Math.PI
					* (spec.getHighFrequency() + 0.5 * spec
							.getTransitionBandwidth()) / spec.fN;
			break;
		case HIGH_PASS:
			w0 = Math.PI;
			w1 = Math.PI
					* (1.0 - (spec.getLowFrequency() - 0.5 * spec
							.getTransitionBandwidth())
							/ spec.fN);
			break;
		case BAND_PASS:
			w0 = 0.5 * Math.PI
					* (spec.getLowFrequency() + spec.getHighFrequency())
					/ spec.fN;
			w1 = 0.5
					* Math.PI
					* (spec.getHighFrequency() - spec.getLowFrequency() + spec
							.getTransitionBandwidth()) / spec.fN;
			break;
		}
		// filter coefficients (NB not normalised to unit maximum gain)
		double[] a = new double[spec.order + 1];
		a[0] = w1 / Math.PI;
		for (int n = 1; n <= m; n++) {
			a[n] = Math.sin(n * w1) * Math.cos(n * w0) * win[n] / (n * Math.PI);
		}
		// shift impulse response to make filter causal
		for (int n = m + 1; n <= spec.order; n++) {
			a[n] = a[n - m];
		}
		for (int n = 0; n <= m - 1; n++) {
			a[n] = a[spec.order - n];
		}
		a[m] = w1 / Math.PI;

		// Normalize
		double g = 0;
		for (int i = 1; i < spec.order + 1; i++) {
			g += a[i];
		}
		for (int i = 1; i < spec.order + 1; i++) {
			a[i] /= g;
		}
		// System.out.println("Filter designed") ;
		return new FIRFilter(a);
	}

	private static int estimatedFIROrder(FilterSpecs spec) {
		int o = 2 * (int) ((spec.getAttenuation() - 7.95)
				/ (14.36 * spec.getTransitionBandwidth() / spec.getFN()) + 1.0);
		return o;
	}

	private static final double sqr(double x) {
		return x * x;
	}

	/**
	 * Calculate the zero order Bessel function of the first kind
	 */
	private static double I0(double x) {
		double eps = 1.0e-6; // accuracy parameter
		double fact = 1.0;
		double x2 = 0.5 * x;
		double p = x2;
		double t = p * p;
		double s = 1.0 + t;
		for (int k = 2; t > eps; k++) {
			p *= x2;
			fact *= k;
			t = sqr(p / fact);
			s += t;
		}
		return s;
	}

	public static final class FilterSpecs {
		FilterType type = FilterType.LOW_PASS;

		int order = -1; // estimated by design()

		double lowFrequency = 0.0f; // low frequency

		double highFrequency = 10000.0f; // high frequency

		double transitionBandwidth = 5000.0f; // transition bandwidth

		double attenuation = 60.0f;

		double fN = 150; // Nyquist, rate/2

		public double getAttenuation() {
			return attenuation;
		}

		public FilterSpecs setAttenuation(double attenuation) {
			this.attenuation = attenuation;
			return this;
		}

		public double getFN() {
			return fN;
		}

		public FilterSpecs setFN(double fn) {
			fN = fn;
			return this;
		}

		public double getHighFrequency() {
			return highFrequency;
		}

		public FilterSpecs setHighFrequency(double highFrequency) {
			this.highFrequency = highFrequency;
			return this;
		}

		public double getLowFrequency() {
			return lowFrequency;
		}

		public FilterSpecs setLowFrequency(double lowFrequency) {
			this.lowFrequency = lowFrequency;
			return this;
		}

		public int getOrder() {
			return order;
		}

		public FilterSpecs setOrder(int order) {
			this.order = order;
			return this;
		}

		public double getTransitionBandwidth() {
			return transitionBandwidth;
		}

		public FilterSpecs setTransitionBandwidth(double transitionBandwidth) {
			this.transitionBandwidth = transitionBandwidth;
			return this;
		}

		public FilterType getType() {
			return type;
		}

		public FilterSpecs setType(FilterType type) {
			this.type = type;
			return this;
		}

	}

	public static void main(String[] args) {
		FIRFilter f = getHighPassFIR(300, 30, 60);
		f.print();
		double[] sig = new double[1000];
		double[] sigf = new double[1000];
		for (int i = 0; i < 1000; i++) {
			sig[i] = Math.cos(2 * Math.PI * 10 * (i / 300.))
					+ Math.cos(2 * Math.PI * 140 * (i / 300));
			sigf[i] = f.apply(sig[i]);
		}
		for (int i = 0; i < 980; i++) {
			System.out.println(sig[i] + " " + sigf[i + 20] + " "
					+ Math.cos(2 * Math.PI * 10 * (i / 300.)));
		}

	}

}
