package org.lsst.ccs.utilities.scheduler;

import java.util.concurrent.CancellationException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Represents a periodic task submitted to a {@link Scheduler}.
 * Can be used to control the task execution.
 *
 * <p><b>Sample Usage.</b> The following code sketch demonstrates explicit construction
 * of a periodic task.
 *
 *  <pre> {@code
 * class Agent {
 * 
 *   private Scheduler scheduler;
 *   private PeriodicTask heartbeat;
 *   ...
 * 
 *   Agent() {
 *     scheduler = new Scheduler("tester", 3);
 *     heartbeat = new PeriodicTask(scheduler, this::broadcast, true, "heartbeat", Level.SEVERE, 5, TimeUnit.SECONDS);
 *     ...
 *   }
 * 
 *   public void start() {
 *     ...
 *     heartbeat.start();
 *   }
 * 
 *   ... 
 * }}</pre>
 * 
 * See {@link Scheduler} documentation for an example of creating an instance of
 * {@code PeriodicTask} by submitting a {@code Runnuble} to a {@link Scheduler}.
 *
 * @author onoprien
 */
public class PeriodicTask implements Runnable, ScheduledFuture<Void> {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final ScheduledThreadPoolExecutor LONG_TASK_DETECTOR
            = new ScheduledThreadPoolExecutor(2, new BasicThreadFactory("Scheduler long task detector", null, true));
    static private final SuspendedTask STOPPED = new SuspendedTask();
    
    private final Scheduler scheduler;
    private final Runnable runner;
    private final boolean isFixedRate; // distinguish between fixed rate and fixed delay tasks
    private final String name;
    private final PeriodicTaskExceptionHandler exceptionHandler;
    
    private volatile ScheduledFuture<?> delegate;
    
    private long period; // nanoseconds; 0 means periodic task is not running
    private boolean isStopped;
    private volatile int failures;
    private int skipped;
    
    private final CountDownLatch cancelLatch = new CountDownLatch(1);
    
    volatile Thread thread;
    volatile long timeBegin,
         timeBeforeOnSkippedExecutions, timeAfterOnSkippedExecutions,
         timeBeforeTimer, timeAfterTimer,
         timeBeforeRun, timeAfterRun,
         timeBeforeCancel,
         timeEnd,
         timeFirstSkipped, delayFirstSkipped;
    
// -- Construction : -----------------------------------------------------------

    /**
     * Creates a periodic task.
     * The task will not be submitted for execution until one of its {@code start(...)} 
     * methods is called.
     * 
     * @param scheduler {@code Scheduler} where this task will run
     * @param runnable {@code Runnable} to execute
     * @param isFixedRate {@code true} if this task will be executed at fixed rate; 
     *                    {@code false} if this task will be run with fixed delay
     * @param taskName Name of this task;
     *                 if {@code null}, no name will be used.
     * @param exceptionHandler Handler that defines this task response to any abnormal circumstances;
     *                         if {@code null}, the default handler will be used.
     * @param period Period between successive executions, or the delay between executions,
     *               depending on the value of {@code isFixedRate} argument.
     * @param unit Time unit of the period parameter.
     * @throws NullPointerException if {@code scheduler} or {@code runnable} arguments are {@code null}
     */
    public PeriodicTask(Scheduler scheduler, Runnable runnable, boolean isFixedRate, String taskName, long period, TimeUnit unit, PeriodicTaskExceptionHandler exceptionHandler) {
        
        if (scheduler == null || runnable == null) throw new NullPointerException();
        
        this.scheduler = scheduler;
        this.runner = runnable;
        this.isFixedRate = isFixedRate;
        this.name = taskName == null ? "" : taskName;
        this.exceptionHandler = exceptionHandler;
        if (exceptionHandler.getLevel() == null) {
            exceptionHandler.setLevel(scheduler.getDefaultLogLevel());
        }
        
        this.period = period > 0L ? unit.toNanos(period) : 0L;
        isStopped = true;
        delegate = STOPPED;
        
        if (exceptionHandler.getMaxFailures() == -1) {
            exceptionHandler.setMaxFailures(scheduler.maxFailures);
        }
    }
    
    public PeriodicTask(Scheduler scheduler, Runnable runnable, boolean isFixedRate, String taskName, Level logLevel, long period, TimeUnit unit) {
        
        if (scheduler == null || runnable == null) throw new NullPointerException();
        
        this.scheduler = scheduler;
        this.runner = runnable;
        this.isFixedRate = isFixedRate;
        this.name = taskName == null ? "" : taskName;
        exceptionHandler = new PeriodicTaskExceptionHandler();
        exceptionHandler.setLevel(logLevel == null ? scheduler.getDefaultLogLevel() : logLevel);
        
        this.period = period > 0L ? unit.toNanos(period) : 0L;
        isStopped = true;
        delegate = STOPPED;
        
        if (exceptionHandler.getMaxFailures() == -1) {
            exceptionHandler.setMaxFailures(scheduler.maxFailures);
        }
    }
    
    
// -- Runnable implementation : ------------------------------------------------

    @Override
    public void run() {
        long time = System.currentTimeMillis();
        thread = Thread.currentThread();
        long timeB = 0L;
        long timeA = 0L;
        if (isFixedRate && exceptionHandler.isSkipOverdueExecutions()) {
            try {
                long d = getDelay(TimeUnit.NANOSECONDS);
                if (d < - period/2) {
                    skipped++;
                    if (skipped == 1) {
                        timeFirstSkipped  = System.currentTimeMillis();
                        delayFirstSkipped = d/1000000;
                    }
                    return;
                } else if (skipped > 0) {
                    timeB = System.currentTimeMillis();
                    exceptionHandler.onSkippedExecutions(this, skipped);
                    timeA = System.currentTimeMillis();
                    skipped = 0;
                }
            } catch (Throwable t) {
            }
        }
        timeBegin = time;
        timeBeforeOnSkippedExecutions = timeB;
        timeAfterOnSkippedExecutions = timeA;
        ScheduledFuture longRunDetector = null;
        try {
            if (exceptionHandler.isDetectLongExecutions()) {
                timeBeforeTimer = System.currentTimeMillis();
                int threshold = exceptionHandler.getLongExecutionThreshold();
                if (threshold > 0) {
                    longRunDetector = LONG_TASK_DETECTOR.schedule(() -> exceptionHandler.onLongExecution(this), threshold, TimeUnit.MILLISECONDS);
                } else {
                    longRunDetector = LONG_TASK_DETECTOR.schedule(() -> exceptionHandler.onLongExecution(this), period*(-threshold), TimeUnit.NANOSECONDS);
                }
                timeAfterTimer = System.currentTimeMillis();
            }
//            try {Thread.currentThread().setName(scheduler.getName() + ":" + name);} catch (RuntimeException x) {}
            timeBeforeRun = System.currentTimeMillis();
            runner.run();
            timeAfterRun = System.currentTimeMillis();
            if (exceptionHandler.isResetFailureCountOnSuccess()) {
                failures = 0;
            }
        } catch (Throwable t) {
            if (exceptionHandler.getMaxFailures() < 0 || ++failures < exceptionHandler.getMaxFailures()) {
                exceptionHandler.onException(this, t);
            } else {
                exceptionHandler.onFinalException(this, t);
                throw t;
            }
        } finally {
//            try {Thread.currentThread().setName(scheduler.getName());} catch (Throwable t) {}
            try {
                if (longRunDetector != null) {
                    timeBeforeCancel = System.currentTimeMillis();
                    longRunDetector.cancel(false);
                }
            } catch (Throwable t) {
            }
            timeEnd = System.currentTimeMillis();
            thread = null;
        }
    }

    /**
     * Get the number of failures that occurred in this Periodic task.
     * @return the number of failures so far.
     */
    public int getFailures() {
        return failures;
    }
    
    /**
     * Get the maximum number of consecutive failures for this task.
     * @return the maximum number of failures for this task.
     * 
     */
    public int getMaxFailures() {
        return exceptionHandler.getMaxFailures();
    }
    
    /**
     * Returns the thread currently executing this task.
     * This method can be called on any thread, but there is no guarantee the returned thread
     * will still be running this task by the time it returns. This method is intended for
     * troubleshooting long-running tasks.
     * 
     * @return Thread currently executing this task.
     */
    public Thread getThread() {
        return thread;
    }

// -- ScheduledFuture implementation : -----------------------------------------
        
    /**
     * Returns the remaining delay associated with this task, in the given time unit.
     *
     * @param unit the time unit
     * @return the remaining delay; zero or negative values indicate that the delay has already elapsed
     */
    @Override
    public long getDelay(TimeUnit unit) {
      return delegate.getDelay(unit);
    }

    @Override
    public int compareTo(Delayed o) {
        return delegate.compareTo(o);
    }

    /**
     * Attempts to cancel execution of this task. This attempt will fail if the task has
     * already completed, has already been canceled, or could not be canceled for some other
     * reason. If successful, and this task has not started when {@code cancel} is called,
     * this task should never run.  If the task has already started, then the
     * {@code mayInterruptIfRunning} parameter determines whether the thread executing this
     * task should be interrupted in an attempt to stop the task.
     *
     * <p>After this method returns, subsequent calls to {@link #isDone} will
     * always return {@code true}.  Subsequent calls to {@link #isCancelled}
     * will always return {@code true} if this method returned {@code true}.
     *
     * @param mayInterruptIfRunning {@code true} if the thread executing this
     *        task should be interrupted; otherwise, in-progress tasks are allowed to complete
     * @return {@code false} if the task could not be canceled,
     *         typically because it has already completed normally; {@code true} otherwise
     */
    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        cancelLatch.countDown();
        return delegate.cancel(mayInterruptIfRunning);
    }

    /** Returns {@code true} if this task has been canceled. */
    @Override
    public boolean isCancelled() {
        return cancelLatch.getCount() == 0;
    }

    /**
     * Returns {@code true} if this task has completed.
     * Completion may be due to normal termination, an exception, or cancellation -
     * in all of these cases, this method will return {@code true}.
     */
    @Override
    public boolean isDone() {
        return cancelLatch.getCount() == 0 || delegate.isDone();
    }

    /**
     * Blocks until the task is canceled.
     *
     * @return {@code null}
     * @throws CancellationException if the computation was canceled
     * @throws ExecutionException if the computation threw an exception
     * @throws InterruptedException if the current thread was interrupted while waiting
     */
    @Override
    public Void get() throws InterruptedException, ExecutionException {
        cancelLatch.await();
        delegate.get();
        return null;
    }

    /**
     * Blocks until the task is canceled or the timeout expires.
     *
     * @param timeout the maximum time to wait
     * @param unit the time unit of the timeout argument
     * @return {@code null}
     * @throws CancellationException if the computation was canceled
     * @throws ExecutionException if the computation threw an exception
     * @throws InterruptedException if the current thread was interrupted while waiting
     * @throws TimeoutException if the wait timed out
     */
    @Override
    public Void get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        long deadline = unit.toNanos(timeout) + System.nanoTime();
        cancelLatch.await(timeout, unit);
        delegate.get(deadline - System.nanoTime(), TimeUnit.NANOSECONDS);
        return null;
    }


// -- Extra functionality : ----------------------------------------------------

    /**
     * Sets the maximum number of consecutive failures (uncaught exceptions) this task can 
     * encounter before being suppressed. By default, the tasks use the maximum
     * number of failures associated with the scheduler they run on.
     * 
     * @param maxFailures Maximum number of consecutive failures; if negative, the task is never suppressed.
     */
    public void setMaxFailures(int maxFailures) {
        exceptionHandler.setMaxFailures(maxFailures);
    }
    
    /**
     * Resets failure count.
     */
    public void resetFailureCount() {
        failures = 0;
    }

    /**
     * Modifies execution period of this task.
     * Non-positive period means the task will not be executed until the period is modified.
     * 
     * If the period is changed while a task is running, two executions can overlap
     * in time. If this is undesirable, call {@link #stop(long, TimeUnit)} before modifying
     * the period, then call {@link #start()}.
     * 
     * @param period the desired period.
     * @param unit the time unit of the period argument
     */
    synchronized public void setPeriod(long period, TimeUnit unit) {
        delegate.cancel(false);
        delegate = STOPPED;
        if (period > 0L) {
            this.period = unit.toNanos(period);
            if (!isStopped) {
                isStopped = true;
                start();
            }
        } else {
            this.period = 0L;
        }
    }
    
    /**
     * Returns execution period of this task.
     * 
     * @param unit time unit for the returned value
     * @return execution period of this task in specified units.
     */
    synchronized public long getPeriod(TimeUnit unit) {
        return unit.convert(period, TimeUnit.NANOSECONDS);
    }
    
    /**
     * Suspends further executions of this periodic task.
     * Suspended tasks can be resumed with the same period by a call to {@code start()}.
     * This method returns without waiting for the currently running execution of this task
     * to complete. Calling this method on a task that is already suspended has no effect.
     */
    synchronized public void stop() {
        if (!isStopped) {
            isStopped = true;
            delegate.cancel(false);
            delegate = STOPPED;
        }
    }
    
    /**
     * Suspends further executions of this periodic task.
     * Suspended tasks can be resumed with the same period by a call to {@code start()}.
     * This method blocks until the currently running execution of this task is complete, or the
     * timeout has expired. Calling this method on a task that is already suspended has no effect.
     * 
     * @param timeout the maximum time to wait; if 0, will wait indefinitely
     * @param unit the time unit of the timeout argument
     * @throws InterruptedException if the current thread was interrupted while waiting
     * @throws TimeoutException if the wait timed out
     */
    synchronized public void stop(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        if (!isStopped) {
            isStopped = true;
            delegate.cancel(false);
            ScheduledFuture<?> old = delegate;
            delegate = STOPPED;
            try {
                old.get(timeout, unit);
            } catch (ExecutionException x) {
            }
        }
    }

    /**
     * Starts executing this task if it is not already being executed.
     * @return {@code false} if this task was already being executed; {@code true} otherwise
     */
    synchronized public boolean start() {
        return start(0L, TimeUnit.NANOSECONDS);
    }
    
    /**
     * Starts executing this task if it is not already being executed.
     * 
     * @param initialDelay the time to delay first execution
     * @param unit the time unit of the initialDelay parameter
     * @return {@code false} if this task was already being executed; {@code true} otherwise
     */
    synchronized public boolean start(long initialDelay, TimeUnit unit) {
        if (isStopped) {
            isStopped = false;
            if (period > 0L) {
                initialDelay = unit.toNanos(initialDelay);
                if (isFixedRate) {
                    delegate = scheduler.executor.scheduleAtFixedRate(this, initialDelay, period, TimeUnit.NANOSECONDS);
                } else {
                    delegate = scheduler.executor.scheduleWithFixedDelay(this, initialDelay, period, TimeUnit.NANOSECONDS);
                }
                return true;
            }
        }
        return false;
    }
    
    /**
     * Get the name of this PeriodicTask
     * @return the PeriodicTask name
     */
    public String getTaskName() {
        return name;
    }
    
    /**
     * Returns the logger used by this task.
     * @return Logger used by this task, or {@code null} if no logger has been set
     *         on the {@link Scheduler} where this task runs.
     */
    public Logger getLog() {
        return scheduler.log;
    }
    
    /**
     * Returns the logger used by this task.
     * @return Logger used by this task, or {@code null} if no logger has been set
     *         on the {@link Scheduler} where this task runs.
     * @deprecated Use {@link #getLog()} instead.
     */
    @Deprecated
    public org.lsst.ccs.utilities.logging.Logger getLogger() {
        return scheduler.getLogger();
    }

    /**
     * Returns the log level set for reporting abnormal circumstances related to this task.
     * @return Log level.
     */
    public Level getLogLevel() {
        return exceptionHandler.getLevel();
    }
    
// -- Delegate for suspended task : --------------------------------------------
    
    private static class SuspendedTask implements ScheduledFuture<Void> {

        @Override
        public long getDelay(TimeUnit unit) {
            return 0L;
        }

        @Override
        public int compareTo(Delayed o) {
            return 0;
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            return true;
        }

        @Override
        public boolean isCancelled() { // never called
            return true;
        }

        @Override
        public boolean isDone() {
            return false;
        }

        @Override
        public Void get() throws InterruptedException, ExecutionException {
            return null;
        }

        @Override
        public Void get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            return null;
        }
        
    }
    
    
// -- Tests : ------------------------------------------------------------------
    
    static public void main(String... args) {
        System.out.println("Start...");
        long t = System.currentTimeMillis();
        for (int i=0; i<1000000; i++) {
            long tt = System.currentTimeMillis();
            if (tt - t > 2) {
                System.out.println(i +" "+ (tt-t));
                t = System.currentTimeMillis();
            } else {
                t = tt;
            }
        }
        System.out.println("Done");
    }
    
}
