package org.lsst.ccs.utilities.logging;

import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

/**
 * Wrapper around {@code java.util.Logger}, kept for backward compatibility, should not be used in new code.
 * The instances are not cached (so better have only one per package)
 * <BR/>
 * Many of those are intended to replace log4j invocations by just changing the
 * <TT>import</TT> directives. The mapping from log4J Levels is operated this
 * way
 * <PRE>
 * finest -> finest
 * debug-> finer
 * fine-> fine
 * info -> info
 * warn -> warning
 * error -> severe
 * fatal -> severe
 * </PRE>
 * <BR/>
 * Logging methods that return a boolean can be used through <TT>assert</TT>
 * calls (they always return true).
 * <p>
 * Do not use this <TT>Logger</TT> for admin purpose (setting a level, a Filter,
 * a Formatter or a Handler): use the corresponding
 * <TT>java.util.logging.Logger</TT> instead.
 *
 * @author bamade
 */
public class Logger {

    private final java.util.logging.Logger julDelegate;

    private Logger(java.util.logging.Logger delegate) {
        this.julDelegate = delegate;

        LogManager manager = LogManager.getLogManager();
        if (manager != null) {
            manager.addLogger(delegate);
        }
    }

    /**
     * Factory method to obtain an instance.
     * This method returns a new instance every time it is called.
     *
     * @param name usually a package name (top of hierarchy is "" empty String).
     * @return New {@code Logger} instance.
     */
    public static Logger getLogger(String name) {
        java.util.logging.Logger realLogger = java.util.logging.Logger.getLogger(name);
        return new Logger(realLogger);
    }

    /**
     * @return the name of the Logger
     */
    public String getName() {
        return julDelegate.getName();
    }

    /**
     * @return the level of the associated JUL logger
     */
    public Level getLevel() {
        return julDelegate.getLevel();
    }

    public void setLevel (Level level){
        julDelegate.setLevel(level);
    }

    /**
     * @return the parent of the associated JUL logger
     */
    protected java.util.logging.Logger getParent() {
        return julDelegate.getParent();
    }

    /**
     * creates a simple log record. This factory manipulates the caller
     * information (className, method Name) by getting rid of any
     * information coming from a package that contains "logging" (or
     * "log4j").
     * <BR/>
     * note that the LogRecord is automatically marked with a sequenceNumber
     * that can be used by <TT>Handlers</TT>
     * that want to detect duplicate publications.
     *
     * @param level   JUL Level
     * @param message (avoid null values)
     * @return a simple LogRecord
     */
    public LogRecord createLogRecord(Level level, String message) {
        LogRecord res = new LogRecord(level, message);
        //now modifies the method and class calls
        //Throwable throwable = new Throwable();
        //StackTraceElement[] stackTraceElements = throwable.getStackTrace();
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        for (StackTraceElement stackTraceElement : stackTraceElements) {
            String className = stackTraceElement.getClassName();
            if (!className.startsWith("org.lsst.ccs.utilities.logging")) {
                if (!className.startsWith("org.apache.log4j") && !className.contains(".logging.")) {
                    String methodName = stackTraceElement.getMethodName();
                    if (!"getStackTrace".equals(methodName)) {
                        res.setSourceClassName(className);
                        res.setSourceMethodName(methodName);
                        break;
                    }
                }
            }
        }

        return res;
    }

    /**
     * utility method to log a message with a Level
     *
     * @param level    JUL level
     * @param message  (avoid null values)
     * @return true
     */
    protected boolean logMessage(Level level, Object message) {
        julDelegate.log(level, String.valueOf(message));
        return true;
    }

    /**
     * logs a message at FINEST level
     *
     * @param message
     * @return true
     */
    public boolean finest(Object message) {
        return logMessage(Level.FINEST, message);
    }

    /**
     * logs a message at FINE Level
     *
     * @param message
     * @return true
     */
    public boolean debug(Object message) {
        return logMessage(Level.FINER, message);
    }
    
    /**
     * logs a message at FINE Level
     *
     * @param message
     * @return true
     */
    public boolean fine(Object message) {
        return logMessage(Level.FINE, message);
    }

    /**
     * logs a message at INFO Level
     *
     * @param message
     * @return true
     */
    public boolean info(Object message) {
        return logMessage(Level.INFO, message);
    }
    
    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @return true
     */
    public boolean warn(Object message) {
        return logMessage(Level.WARNING, message);
    }
    
    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @return true
     */
    public boolean warning(Object message) {
        return logMessage(Level.WARNING, message);
    }

    /**
     * logs a message at SEVERE Level (use preferably the error method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     */
    public void error(Object message) {
        logMessage(Level.SEVERE, message);
    }

    /**
     * logs a message at SEVERE Level (use preferably the fatal method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     */
    public void fatal(Object message) {
        logMessage(Level.SEVERE, message);
    }
    
    /**
     * logs a message at SEVERE Level (use preferably the severe method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     */
    public void severe(Object message) {
        logMessage(Level.SEVERE, message);
    }

    /**
     * tells if the FINER level is activated for the corresponding JUL
     * Logger. (to be used if a subsequent logging call is costly)
     *
     * @return
     */
    public boolean isDebugEnabled() {
        return julDelegate.isLoggable(Level.FINER);
    }

    /**
     * tells if the INFO level is activated for the corresponding JUL
     * Logger. (to be used if a subsequent logging call is costly)
     *
     * @return
     */
    public boolean isInfoEnabled() {
        return julDelegate.isLoggable(Level.INFO);
    }

    /**
     * utility method to forward a <TT>Throwable</TT> and a message to the loggers.
     *
     * @param level     JUL level
     * @param message
     * @param throwable
    */
    protected void logSimpleThrowable(Level level, Object message, Throwable throwable) {
        julDelegate.log(level, String.valueOf(message), throwable);
    }
    
    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     */
    public void fatal(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }
    
    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     */
    public void severe(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }

    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     */
    public void error(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }

    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param throwable
     */
    public void warn(Object message, Throwable throwable) {
        logSimpleThrowable(Level.WARNING, message, throwable);
    }

    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param throwable
     */
    public void warning(Object message, Throwable throwable) {
        logSimpleThrowable(Level.WARNING, message, throwable);
    }
    
    /**
     * invokes the <TT>throwing</TT> method on the corresponding JUL logger.
     *
     * @param sourceClass
     * @param sourceMethod
     * @param throwable
     */
    public void throwing(String sourceClass, String sourceMethod, Throwable throwable) {
        julDelegate.throwing(sourceClass, sourceMethod, throwable);
    }

    /**
     * logs a message at INFO Level
     *
     * @param message
     * @param throwable
     */
    public void info(Object message, Throwable throwable) {
        logSimpleThrowable(Level.INFO, message, throwable);
    }

    /**
     * logs a message at FINER Level
     *
     * @param message
     * @param throwable
     */
    public void debug(Object message, Throwable throwable) {
        logSimpleThrowable(Level.FINER, message, throwable);
    }

    //END LOG4J
    // BEGIN JUL-LIKE calls

    /**
     * tells if the corresponding JUL level is activated for the
     * corresponding JUL Logger. (to be used if a subsequent logging call is
     * costly)
     *
     * @param level The level for which we are asking if it's loggable
     * @return
     */
    public boolean isLoggable(Level level) {
        return julDelegate.isLoggable(level);
    }

    /**
     * utility method to send a <TT>LogRecord</TT> to the corresponding JUL
     * logger and to a list of other loggers.
     *
     * @param record   (should be created with the <TT>createLogRecord</TT>
     *                 factory :otherwise stack information will be wrong)
     */
    public void log(LogRecord record){
        record.setLoggerName(julDelegate.getName());
        try {
            julDelegate.log(record);
        } catch (Exception e) {
            System.out.println("Problem logging record due to exception: "+StackTraceFormats.toString(e));
        }
    }

    /**
     * Kept for backward compatibility. Equivalent to {@code log(LogRecord)}.
     */
    @Deprecated
    public void decoupledLog(long delay, final LogRecord record) {
        log(record);
    }
    
    /**
     * Does the same as the equivalent standard JUL method.
     *
     * @param level    JUL level
     * @param message
     * @param argument
     * @return true.
     */
    public boolean log(Level level, String message, Object argument) {
        julDelegate.log(level, message, argument);
        return true;
    }

    
    /**
     * Kept for backward compatibility. Equivalent to {@code log(Level, String, Object)}.
     * @return true.
     */
    @Deprecated
    public boolean decoupledLog(long delay, Level level, String message, Object argument) {
        log(level, message, argument);
        return true;
    }

    /**
     * Does the same as the equivalent standard JUL method.
     *
     * @param level     JUL level
     * @param message
     * @param arguments
     * @return true
     */
    public boolean log(Level level, String message, Object[] arguments) {
        julDelegate.log(level, message, arguments);
        return true;
    }

    /**
     * Kept for backward compatibility. Equivalent to {@code log(Level, String, Object[])}.
     * @return true.
     */
    @Deprecated
    public boolean decoupledLog(long delay, Level level, String message, Object[] arguments) {
        log(level, message, arguments);
        return true;
    }

    /**
     * does the same as the equivalent standard JUL method.
     *
     * @param level     JUL level
     * @param message
     * @param throwable
     */
    public void log(Level level, String message, Throwable throwable) {
        julDelegate.log(level, message, throwable);
    }

    /**
     * Kept for backward compatibility. Equivalent to {@code log(Level, String, Throwable)}.
     */
    @Deprecated
    public void decoupledLog(long delay, Level level, String message, Throwable throwable) {
        log(level, message, throwable);
    }


}
